import React, { SVGAttributes, forwardRef } from 'react';

import { useUniqId } from '@use-platform/react';

export interface SafariMobileProps extends SVGAttributes<SVGSVGElement> {
  /**
   * Additional className for svg root
   */
  className?: string;
  /**
   * Icon size
   *
   * @default 24
   */
  size?: 12 | 16 | 24 | 32;
}

export const SafariMobile = forwardRef<SVGSVGElement, SafariMobileProps>((props, ref) => {
  const { className, size = 24, ...otherProps } = props;
  const id = useUniqId();

  return (
    <svg
      width={size}
      height={size}
      viewBox="0 0 24 24"
      fill="none"
      {...otherProps}
      focusable="false"
      aria-hidden="true"
      className={'SvgIcon' + (className ? ' ' + className : '')}
      ref={ref}
    >
      <path
        d="M12 22c5.523 0 10-4.477 10-10S17.523 2 12 2 2 6.477 2 12s4.477 10 10 10Z"
        fill={`url(#${id}1kqfjmj_a)`}
      />
      <path
        d="M12 21.25a9.25 9.25 0 1 0 0-18.5 9.25 9.25 0 0 0 0 18.5Z"
        fill={`url(#${id}1kqfjmj_b)`}
      />
      <path
        d="M12.016 5c-.063 0-.118-.047-.118-.11V3.259c0-.063.055-.11.118-.11.062 0 .117.047.117.11V4.89c-.008.062-.055.109-.117.109ZM12.797 4.125c-.063-.008-.11-.055-.11-.117l.055-.727a.114.114 0 0 1 .125-.101c.063.008.11.054.11.117l-.055.726a.114.114 0 0 1-.125.102ZM13.383 5.14a.12.12 0 0 1-.094-.132l.32-1.602c.016-.062.07-.101.133-.086a.12.12 0 0 1 .094.133l-.32 1.602c-.008.054-.07.094-.133.086ZM14.32 4.438a.117.117 0 0 1-.086-.141l.196-.703c.015-.063.078-.094.14-.078a.116.116 0 0 1 .086.14l-.195.703c-.016.063-.078.094-.14.079ZM14.703 5.54a.112.112 0 0 1-.062-.15l.625-1.507c.023-.055.094-.086.148-.055a.112.112 0 0 1 .063.148l-.625 1.508c-.024.055-.094.086-.149.055ZM15.758 5.04c-.055-.032-.079-.095-.055-.15l.328-.648a.114.114 0 0 1 .156-.047c.055.032.078.094.055.149l-.328.648a.114.114 0 0 1-.156.047ZM15.914 6.195c-.054-.039-.07-.109-.031-.156l.906-1.351c.031-.055.102-.063.156-.032.055.04.07.11.032.157l-.907 1.351a.106.106 0 0 1-.156.031ZM17.055 5.906c-.047-.039-.063-.11-.024-.156l.454-.57a.111.111 0 0 1 .156-.016c.047.04.062.11.023.156l-.453.57a.112.112 0 0 1-.156.016ZM16.977 7.078c-.047-.047-.047-.117 0-.156l1.156-1.148a.11.11 0 0 1 .156 0c.039.039.047.117 0 .156l-1.148 1.148a.113.113 0 0 1-.164 0ZM18.148 7.016c-.039-.047-.039-.118.008-.157l.555-.468a.111.111 0 0 1 .156.015c.04.047.04.117-.008.156l-.554.47a.112.112 0 0 1-.157-.016ZM17.836 8.14a.106.106 0 0 1 .031-.156l1.36-.906c.054-.031.125-.015.156.031a.106.106 0 0 1-.031.157l-1.36.906c-.047.031-.117.016-.156-.031ZM19 8.305a.114.114 0 0 1 .04-.156l.632-.36c.055-.031.125-.008.156.047a.114.114 0 0 1-.039.156l-.633.36c-.054.03-.124.008-.156-.047ZM18.477 9.344c-.024-.063 0-.125.062-.149l1.508-.625c.055-.023.125.008.148.063.024.062 0 .125-.062.148l-1.508.625a.112.112 0 0 1-.148-.062ZM19.578 9.734c-.023-.062.008-.125.07-.148l.688-.227c.055-.023.125.016.14.07.024.063-.007.126-.07.15l-.687.226c-.055.023-.117-.008-.14-.07ZM18.875 10.664c-.016-.062.023-.125.086-.133l1.601-.312c.063-.016.118.031.133.094.016.062-.023.124-.086.132l-1.601.313c-.063.008-.125-.031-.133-.094ZM19.875 11.266c-.008-.063.031-.117.094-.125l.718-.086a.114.114 0 0 1 .125.101c.008.063-.03.117-.093.125l-.72.086c-.054.008-.116-.039-.124-.101ZM19 12.023c0-.062.047-.117.11-.117h1.632c.063 0 .11.055.11.117 0 .063-.047.118-.11.118H19.11c-.062 0-.109-.055-.109-.118ZM19.867 12.813c.008-.063.055-.11.117-.11l.727.055c.062.008.11.062.102.125-.008.062-.055.11-.118.11l-.726-.056c-.055-.007-.102-.062-.102-.124ZM18.86 13.383a.12.12 0 0 1 .132-.094l1.602.32c.062.016.101.07.086.133a.12.12 0 0 1-.133.094l-1.602-.32c-.054-.008-.093-.07-.086-.133ZM19.563 14.32a.117.117 0 0 1 .14-.085l.703.195c.063.015.094.078.079.14a.117.117 0 0 1-.141.086l-.703-.195c-.063-.016-.094-.078-.078-.14ZM18.461 14.695a.112.112 0 0 1 .149-.062l1.507.625c.055.023.086.094.063.148a.112.112 0 0 1-.149.063l-1.507-.625a.112.112 0 0 1-.063-.149ZM18.969 15.75c.03-.055.093-.078.148-.055l.648.328c.055.024.078.094.047.157-.031.054-.094.078-.148.054l-.649-.328c-.054-.031-.078-.101-.046-.156ZM17.82 15.898c.04-.054.102-.07.157-.03l1.36.905c.054.032.062.102.03.157-.039.054-.101.07-.156.03l-1.36-.905c-.054-.04-.07-.11-.03-.157ZM18.11 17.031c.038-.047.109-.062.156-.023l.57.445c.047.04.055.11.016.157-.04.046-.11.062-.157.023l-.57-.445a.111.111 0 0 1-.015-.157ZM16.945 16.953c.047-.047.118-.047.157 0l1.156 1.149a.11.11 0 0 1 0 .156c-.047.047-.117.047-.156 0l-1.157-1.149c-.039-.039-.039-.117 0-.156ZM17.016 18.117c.047-.039.117-.039.156.008l.477.547a.112.112 0 0 1-.016.156c-.047.04-.117.04-.156-.008L17 18.273c-.039-.038-.031-.109.016-.156ZM15.89 17.82a.106.106 0 0 1 .157.032l.906 1.351c.032.055.016.125-.03.157a.106.106 0 0 1-.157-.032l-.906-1.351c-.032-.055-.016-.125.03-.157ZM15.726 18.977a.114.114 0 0 1 .157.039l.359.632c.031.055.008.125-.047.157a.114.114 0 0 1-.156-.04l-.36-.632c-.023-.055-.007-.125.047-.156ZM14.688 18.46c.062-.023.124 0 .148.056l.633 1.507c.023.055-.008.125-.063.149-.054.023-.125 0-.148-.055l-.633-1.508c-.023-.054.008-.125.063-.148ZM14.305 19.57c.062-.023.125.008.148.07l.227.688c.023.055-.016.125-.07.14-.063.024-.126-.007-.15-.07l-.226-.687c-.023-.055.008-.117.07-.14ZM13.383 18.86c.062-.016.125.023.133.085l.328 1.602c.016.062-.031.117-.094.133-.062.015-.125-.024-.133-.086l-.328-1.602a.125.125 0 0 1 .094-.133ZM12.79 19.875a.12.12 0 0 1 .132.094l.094.718a.114.114 0 0 1-.102.125.12.12 0 0 1-.132-.093L12.688 20a.114.114 0 0 1 .101-.125ZM11.984 19c.063 0 .117.047.117.11v1.632c0 .063-.054.11-.117.11-.062 0-.117-.047-.117-.11V19.11c.008-.062.055-.11.117-.11ZM11.203 19.875c.063.008.11.055.11.117l-.055.727a.114.114 0 0 1-.125.101c-.063-.008-.11-.054-.11-.117l.055-.726a.114.114 0 0 1 .125-.102ZM10.617 18.86a.12.12 0 0 1 .094.132l-.32 1.602c-.016.062-.07.101-.133.086a.12.12 0 0 1-.094-.133l.32-1.602c.008-.054.07-.093.133-.086ZM9.68 19.563a.116.116 0 0 1 .086.14l-.195.703c-.016.063-.078.094-.14.078a.117.117 0 0 1-.087-.14l.195-.703c.016-.063.078-.094.141-.078ZM9.297 18.46a.112.112 0 0 1 .062.15l-.625 1.507c-.023.055-.093.086-.148.055-.055-.031-.086-.094-.062-.149l.624-1.507c.024-.055.094-.086.149-.055ZM8.242 18.961c.055.031.078.094.055.148l-.328.649a.114.114 0 0 1-.157.047c-.054-.032-.078-.094-.054-.149l.328-.648a.114.114 0 0 1 .156-.047ZM8.086 17.805c.055.039.07.11.031.156l-.906 1.352c-.031.054-.101.062-.156.03-.055-.038-.07-.108-.031-.155l.906-1.352a.106.106 0 0 1 .156-.031ZM6.945 18.094c.047.039.063.11.024.156l-.453.57a.111.111 0 0 1-.156.016c-.047-.04-.063-.11-.024-.156l.453-.57a.111.111 0 0 1 .156-.016ZM7.023 16.922c.047.047.047.117 0 .156l-1.156 1.149a.11.11 0 0 1-.156 0c-.04-.04-.047-.118 0-.157l1.148-1.148a.113.113 0 0 1 .164 0ZM5.852 16.984c.039.047.039.118-.008.157l-.555.469a.112.112 0 0 1-.156-.016c-.04-.047-.04-.117.008-.157l.554-.468a.111.111 0 0 1 .157.015ZM6.164 15.86a.106.106 0 0 1-.031.156l-1.36.906c-.054.031-.124.015-.156-.031a.106.106 0 0 1 .032-.157l1.36-.906c.046-.031.116-.015.155.031ZM5 15.695a.114.114 0 0 1-.039.157l-.633.359c-.054.031-.125.008-.156-.047a.114.114 0 0 1 .04-.156l.632-.36c.055-.03.125-.007.156.047ZM5.524 14.656c.023.063 0 .125-.063.149l-1.508.625c-.054.023-.125-.008-.148-.063-.024-.062 0-.125.062-.148l1.508-.625a.112.112 0 0 1 .149.062ZM4.422 14.266c.023.062-.008.125-.07.148l-.688.227c-.055.023-.125-.016-.14-.07-.024-.063.007-.126.07-.149l.687-.227c.055-.023.117.008.14.07ZM5.125 13.336c.016.063-.023.125-.086.133l-1.602.312c-.062.016-.117-.031-.132-.094-.016-.062.023-.124.086-.132l1.601-.313c.063-.008.125.031.133.094ZM4.125 12.734c.008.063-.031.118-.094.126l-.718.085a.114.114 0 0 1-.125-.101.108.108 0 0 1 .093-.125L4 12.633c.055-.008.117.039.125.101ZM5 11.976c0 .063-.047.118-.11.118H3.259c-.063 0-.11-.055-.11-.117 0-.063.047-.118.11-.118H4.89c.062 0 .109.055.109.117ZM4.133 11.188c-.008.062-.055.109-.117.109l-.727-.055a.114.114 0 0 1-.102-.125c.008-.062.055-.11.118-.11l.726.056c.055.007.102.062.102.124ZM5.14 10.617a.12.12 0 0 1-.132.094l-1.602-.328c-.062-.016-.101-.07-.086-.133a.12.12 0 0 1 .133-.094l1.602.32c.054.016.093.079.086.141ZM4.438 9.68a.116.116 0 0 1-.141.086l-.703-.196c-.063-.015-.094-.078-.078-.14a.116.116 0 0 1 .14-.086l.703.195c.063.016.094.078.079.14ZM5.54 9.305a.112.112 0 0 1-.15.062l-1.507-.625c-.055-.023-.086-.094-.063-.148a.112.112 0 0 1 .149-.063l1.508.625a.112.112 0 0 1 .062.149ZM5.032 8.25c-.032.055-.094.078-.149.055l-.648-.328c-.055-.024-.078-.094-.047-.157.031-.054.094-.078.148-.054l.649.328c.054.031.078.101.047.156ZM6.18 8.101c-.04.055-.102.07-.157.032l-1.351-.899c-.055-.03-.063-.101-.032-.156a.106.106 0 0 1 .157-.031l1.359.906c.047.031.063.102.023.148ZM5.89 6.969c-.038.047-.109.062-.155.023l-.57-.445a.112.112 0 0 1-.016-.156c.039-.047.109-.063.156-.024l.57.446c.047.039.055.109.016.156ZM7.055 7.047c-.047.047-.118.047-.157 0L5.734 5.899a.11.11 0 0 1 0-.157c.047-.047.118-.047.157 0l1.156 1.149a.111.111 0 0 1 .008.156ZM6.984 5.883c-.046.039-.117.039-.156-.008l-.476-.547a.111.111 0 0 1 .015-.156c.047-.04.117-.04.157.008L7 5.727c.04.039.031.109-.016.156ZM8.11 6.18a.106.106 0 0 1-.157-.031L7.04 4.796c-.031-.055-.015-.125.031-.156a.106.106 0 0 1 .157.031l.906 1.352a.11.11 0 0 1-.023.156ZM8.273 5.024a.114.114 0 0 1-.156-.04l-.36-.632c-.03-.055-.007-.125.048-.157a.114.114 0 0 1 .156.04l.36.632c.023.055.007.125-.048.157ZM9.312 5.539c-.062.023-.125 0-.148-.055l-.633-1.508c-.023-.054.008-.125.063-.148.062-.023.125 0 .148.055l.633 1.508c.023.054-.008.125-.063.148ZM9.695 4.43c-.062.023-.125-.008-.148-.07l-.227-.688c-.023-.055.016-.125.07-.14.055-.016.126.007.15.07l.226.687c.023.055-.008.117-.07.14ZM10.617 5.14c-.062.016-.125-.023-.133-.085l-.328-1.602c-.015-.062.032-.117.094-.133.063-.015.125.024.133.086l.328 1.602a.125.125 0 0 1-.094.133ZM11.21 4.125a.12.12 0 0 1-.132-.094l-.094-.718a.114.114 0 0 1 .102-.125.12.12 0 0 1 .133.093l.093.719a.114.114 0 0 1-.101.125Z"
        fill="#F3F3F3"
      />
      <path
        d="M12.016 5.008c-.07 0-.125-.055-.125-.117V3.258c0-.063.054-.117.125-.117.07 0 .125.054.125.117V4.89c0 .062-.063.117-.125.117Zm0-1.852c-.055 0-.11.047-.11.102V4.89c0 .054.047.101.11.101.054 0 .109-.047.109-.101V3.258c-.008-.055-.055-.102-.11-.102ZM12.797 4.133a.127.127 0 0 1-.117-.125l.054-.727c.008-.062.063-.117.133-.11.07.009.117.063.117.126l-.054.726c-.008.07-.063.118-.133.11Zm.07-.945c-.055 0-.11.039-.11.093l-.054.727c0 .055.04.101.102.11.054 0 .109-.04.109-.095l.055-.726c0-.055-.047-.102-.102-.11ZM13.383 5.148a.127.127 0 0 1-.102-.14l.32-1.602a.126.126 0 0 1 .149-.094c.07.016.11.079.101.141l-.32 1.602c-.015.062-.078.101-.148.093Zm.367-1.82a.112.112 0 0 0-.125.078l-.32 1.602a.11.11 0 0 0 .086.125.112.112 0 0 0 .125-.078l.32-1.602c.008-.062-.031-.11-.086-.125ZM14.32 4.445c-.062-.015-.109-.086-.085-.148l.195-.703c.015-.063.086-.102.148-.086.063.015.11.086.086.148l-.195.703c-.016.063-.078.102-.149.086Zm.258-.914c-.054-.015-.117.016-.133.07l-.195.704c-.015.054.016.11.078.125.055.015.117-.016.133-.07l.195-.704a.112.112 0 0 0-.078-.125ZM14.703 5.547a.121.121 0 0 1-.07-.156l.625-1.508c.023-.063.094-.086.164-.063a.121.121 0 0 1 .07.157l-.625 1.507c-.031.063-.101.094-.164.063Zm.711-1.711a.106.106 0 0 0-.14.055l-.625 1.507c-.024.055.007.11.062.133.055.024.117 0 .14-.054l.626-1.508a.111.111 0 0 0-.063-.133ZM15.758 5.047a.12.12 0 0 1-.055-.164l.328-.649a.12.12 0 0 1 .164-.054.12.12 0 0 1 .055.164l-.328.648a.12.12 0 0 1-.164.055Zm.43-.852c-.055-.023-.118-.008-.141.047l-.328.649c-.024.046 0 .109.047.14.054.024.117.008.14-.047l.329-.648c.023-.047 0-.11-.047-.14ZM15.914 6.203a.135.135 0 0 1-.04-.172l.907-1.351a.123.123 0 0 1 .172-.031c.055.039.07.117.039.171l-.906 1.352a.125.125 0 0 1-.172.031Zm1.031-1.539c-.047-.031-.117-.023-.148.024l-.907 1.351a.102.102 0 0 0 .032.14c.046.032.117.024.148-.023l.906-1.343a.111.111 0 0 0-.03-.149ZM17.047 5.914c-.055-.039-.063-.117-.024-.172l.454-.57c.038-.055.117-.063.171-.016.055.04.063.117.024.172l-.453.57c-.04.055-.117.063-.172.016Zm.586-.742c-.047-.04-.11-.031-.149.016l-.453.57a.11.11 0 0 0 .024.148c.047.04.109.032.148-.015l.453-.57c.032-.048.024-.118-.023-.15ZM16.969 7.086a.122.122 0 0 1 0-.172l1.156-1.148a.123.123 0 0 1 .172 0 .122.122 0 0 1 0 .172L17.14 7.086a.123.123 0 0 1-.172 0Zm1.32-1.313a.109.109 0 0 0-.148 0l-1.157 1.149c-.039.039-.039.11 0 .148.04.04.11.04.149 0l1.156-1.148c.04-.04.04-.102 0-.149ZM18.14 7.023c-.046-.054-.038-.132.009-.171l.554-.47c.047-.046.125-.038.172.016.047.055.04.133-.008.172l-.554.47c-.047.046-.125.038-.172-.017Zm.72-.617c-.04-.047-.102-.047-.149-.015l-.554.468a.109.109 0 0 0-.008.149c.039.047.101.047.148.015l.555-.468c.047-.04.047-.102.008-.149ZM17.828 8.14a.125.125 0 0 1 .032-.171l1.359-.906a.12.12 0 0 1 .172.039.125.125 0 0 1-.031.171L18 8.18a.12.12 0 0 1-.172-.04Zm1.547-1.023a.102.102 0 0 0-.14-.031l-1.36.906c-.047.031-.062.094-.023.149a.102.102 0 0 0 .14.031l1.36-.906a.11.11 0 0 0 .023-.149ZM18.992 8.313c-.03-.063-.015-.133.04-.164l.632-.36c.055-.031.133-.008.164.047.032.063.016.133-.039.164l-.633.36c-.054.03-.132.007-.163-.047Zm.829-.47c-.032-.054-.094-.07-.141-.038l-.633.36a.112.112 0 0 0-.039.14c.031.054.094.07.14.039l.634-.36c.046-.023.062-.085.039-.14ZM18.469 9.351a.13.13 0 0 1 .062-.164l1.508-.625a.121.121 0 0 1 .156.07.13.13 0 0 1-.062.165l-1.508.625a.121.121 0 0 1-.156-.07Zm1.71-.71a.111.111 0 0 0-.132-.063l-1.508.625c-.055.023-.078.086-.055.14a.111.111 0 0 0 .133.063l1.508-.625c.055-.023.078-.086.055-.14ZM19.57 9.742a.12.12 0 0 1 .078-.156l.688-.227c.062-.023.133.016.156.079a.12.12 0 0 1-.078.156l-.688.226a.133.133 0 0 1-.156-.078Zm.906-.297c-.015-.054-.078-.086-.132-.07l-.688.227c-.054.015-.078.078-.062.132.015.055.078.086.132.07l.688-.226a.108.108 0 0 0 .062-.133ZM18.86 10.664a.126.126 0 0 1 .093-.148l1.602-.313c.062-.015.125.032.14.102a.126.126 0 0 1-.093.148L19 10.766c-.062.015-.125-.031-.14-.102Zm1.82-.351a.105.105 0 0 0-.117-.086l-1.602.312c-.055.008-.094.07-.078.125a.105.105 0 0 0 .117.086l1.602-.312c.054-.016.093-.07.078-.125ZM19.867 11.266c-.008-.07.04-.133.102-.141l.718-.086a.121.121 0 0 1 .133.11c.008.07-.039.132-.101.14l-.72.086a.121.121 0 0 1-.132-.11Zm.945-.11c-.008-.054-.062-.101-.117-.094l-.719.086a.105.105 0 0 0-.085.118c.007.054.062.101.117.093l.718-.086a.105.105 0 0 0 .086-.117ZM18.992 12.023c0-.07.055-.125.117-.125h1.633c.063 0 .117.055.117.125s-.054.125-.117.125H19.11c-.062 0-.117-.054-.117-.125Zm1.852.008c0-.054-.047-.11-.102-.11H19.11c-.054 0-.101.048-.101.11 0 .055.047.11.101.11h1.633c.055-.008.102-.055.102-.11ZM19.86 12.813a.127.127 0 0 1 .124-.118l.727.055c.063.008.117.063.11.133a.127.127 0 0 1-.126.117l-.726-.055c-.063-.008-.11-.07-.11-.133Zm.953.07a.099.099 0 0 0-.094-.11l-.727-.054c-.055 0-.101.039-.11.101-.007.055.04.11.095.11l.726.054c.055 0 .102-.046.11-.101ZM18.852 13.383a.128.128 0 0 1 .14-.102l1.602.32c.062.016.11.079.094.149a.127.127 0 0 1-.141.102l-1.602-.32c-.062-.016-.101-.079-.093-.15Zm1.82.367a.112.112 0 0 0-.078-.125l-1.602-.32a.111.111 0 0 0-.125.086.112.112 0 0 0 .078.125l1.602.32c.063.008.11-.031.125-.086ZM19.555 14.32c.015-.062.086-.109.148-.086l.703.196a.125.125 0 0 1 .086.148c-.015.063-.086.11-.148.086l-.703-.195c-.063-.016-.102-.079-.086-.149Zm.914.258c.015-.055-.016-.117-.07-.133l-.704-.195c-.054-.016-.109.016-.125.078-.015.055.016.117.07.133l.704.195a.112.112 0 0 0 .125-.078ZM18.453 14.688a.121.121 0 0 1 .157-.07l1.507.624c.063.024.086.094.063.164a.121.121 0 0 1-.157.07l-1.507-.624a.13.13 0 0 1-.063-.165Zm1.711.71a.106.106 0 0 0-.055-.14l-1.507-.625c-.055-.023-.11.008-.133.062a.106.106 0 0 0 .055.141l1.507.625c.055.016.117-.008.133-.063ZM18.961 15.742a.12.12 0 0 1 .164-.055l.649.329a.12.12 0 0 1 .054.164.12.12 0 0 1-.164.054l-.648-.328c-.063-.023-.086-.101-.055-.164Zm.844.43c.023-.055.008-.117-.047-.14l-.648-.329c-.047-.023-.11 0-.141.047-.024.055-.008.117.047.14l.648.329c.055.023.117.008.14-.047ZM17.812 15.89a.135.135 0 0 1 .172-.038l1.36.906c.054.039.07.11.031.172a.135.135 0 0 1-.172.039l-1.36-.907c-.062-.039-.07-.117-.03-.171Zm1.54 1.032c.03-.047.023-.117-.024-.149l-1.36-.906a.102.102 0 0 0-.14.031c-.031.047-.023.118.024.149l1.359.906c.047.031.11.016.14-.031ZM18.102 17.023c.039-.054.117-.062.171-.023l.57.445c.055.04.063.117.024.172-.039.055-.117.063-.172.024l-.57-.446c-.055-.039-.063-.117-.023-.172Zm.75.586c.039-.047.03-.109-.016-.148l-.57-.445a.11.11 0 0 0-.149.023c-.039.047-.031.11.016.148l.57.446a.11.11 0 0 0 .149-.024ZM16.945 16.945a.122.122 0 0 1 .172 0l1.156 1.149a.122.122 0 0 1 0 .172.122.122 0 0 1-.171 0l-1.157-1.149a.11.11 0 0 1 0-.172Zm1.313 1.305a.109.109 0 0 0 0-.148l-1.156-1.149a.109.109 0 0 0-.149 0 .109.109 0 0 0 0 .149l1.157 1.148c.038.047.101.047.148 0ZM17.008 18.117a.123.123 0 0 1 .172.008l.477.547c.046.047.039.125-.016.172a.123.123 0 0 1-.172-.008l-.477-.547c-.039-.055-.039-.133.016-.172Zm.625.711c.047-.039.047-.101.016-.148l-.477-.547a.108.108 0 0 0-.148-.008c-.047.04-.047.102-.016.148l.477.547c.039.047.101.047.148.008ZM15.89 17.813a.125.125 0 0 1 .173.03l.906 1.352a.12.12 0 0 1-.04.172.125.125 0 0 1-.171-.031l-.906-1.352a.119.119 0 0 1 .039-.171Zm1.032 1.539a.102.102 0 0 0 .031-.141l-.906-1.352c-.031-.046-.094-.054-.148-.023a.102.102 0 0 0-.031.14l.906 1.352c.03.04.101.055.148.024ZM15.727 18.969c.062-.032.132-.016.164.039l.359.633c.031.054.016.133-.047.164-.062.031-.133.015-.164-.04l-.36-.632c-.03-.055-.015-.125.048-.164Zm.468.828c.047-.031.07-.094.04-.14l-.36-.634a.112.112 0 0 0-.14-.039c-.047.032-.07.094-.04.141l.36.633c.023.047.093.07.14.04ZM14.687 18.453a.13.13 0 0 1 .165.063l.632 1.507a.122.122 0 0 1-.07.157.13.13 0 0 1-.164-.063l-.633-1.508c-.023-.054.008-.125.07-.156Zm.711 1.711c.055-.023.079-.086.055-.133l-.633-1.508a.106.106 0 0 0-.14-.054c-.055.023-.079.086-.055.133l.633 1.507c.031.055.086.078.14.055ZM14.297 19.563a.121.121 0 0 1 .156.07l.227.687c.023.063-.016.133-.078.157a.121.121 0 0 1-.157-.07l-.226-.688a.133.133 0 0 1 .078-.156Zm.305.898c.054-.016.086-.078.07-.133l-.227-.687c-.015-.055-.078-.078-.132-.063-.055.016-.086.078-.07.133l.226.688a.098.098 0 0 0 .133.062ZM13.383 18.851a.126.126 0 0 1 .148.094l.328 1.602c.016.062-.03.133-.093.14a.126.126 0 0 1-.149-.093l-.328-1.602c-.015-.063.024-.125.094-.14Zm.367 1.82c.055-.007.094-.062.086-.124l-.328-1.602c-.008-.055-.07-.093-.125-.078-.055.008-.094.063-.086.125l.328 1.602c.016.054.07.086.125.078ZM12.789 19.867c.07-.008.133.04.14.102l.094.718a.122.122 0 0 1-.109.133c-.07.008-.133-.039-.14-.101L12.68 20a.121.121 0 0 1 .109-.133Zm.117.938c.055-.008.102-.063.094-.117l-.094-.72a.105.105 0 0 0-.117-.085c-.055.008-.102.062-.094.117l.094.719a.105.105 0 0 0 .117.086ZM11.984 18.992c.07 0 .125.055.125.117v1.633c0 .063-.054.117-.125.117-.07 0-.125-.054-.125-.117V19.11c0-.062.063-.117.125-.117Zm0 1.852c.055 0 .11-.047.11-.102V19.11c0-.054-.047-.101-.11-.101-.054 0-.109.047-.109.101v1.633c.008.055.055.102.11.102ZM11.203 19.867c.07.008.117.063.117.125l-.054.727c-.008.062-.063.117-.133.11a.127.127 0 0 1-.117-.126l.054-.726c.008-.07.063-.118.133-.11Zm-.07.945c.055 0 .11-.039.11-.093l.054-.727c0-.055-.04-.101-.102-.11-.054 0-.11.04-.11.095l-.054.726c0 .055.047.102.102.11ZM10.617 18.851c.07.016.11.079.102.141l-.32 1.602a.126.126 0 0 1-.149.093.127.127 0 0 1-.102-.14l.32-1.602c.016-.062.079-.102.15-.094Zm-.367 1.82a.112.112 0 0 0 .125-.077l.32-1.602a.11.11 0 0 0-.086-.125.112.112 0 0 0-.125.078l-.32 1.602c-.008.062.031.11.086.125ZM9.68 19.555c.062.015.11.086.086.148l-.195.703a.125.125 0 0 1-.149.086c-.062-.015-.11-.086-.086-.148l.196-.703c.015-.063.078-.102.148-.086Zm-.258.914c.055.015.117-.016.133-.07l.195-.704c.016-.054-.015-.109-.078-.125-.055-.015-.117.016-.133.07l-.195.704a.112.112 0 0 0 .078.125ZM9.297 18.453a.121.121 0 0 1 .07.156l-.625 1.508c-.023.063-.094.086-.164.063a.122.122 0 0 1-.07-.157l.625-1.507c.031-.063.101-.094.164-.063Zm-.711 1.711c.055.023.117 0 .14-.055l.626-1.507c.023-.055-.008-.11-.063-.133a.106.106 0 0 0-.14.054l-.626 1.508a.111.111 0 0 0 .063.133ZM8.242 18.953a.12.12 0 0 1 .055.164l-.328.649a.12.12 0 0 1-.164.054.12.12 0 0 1-.055-.164l.328-.648a.12.12 0 0 1 .164-.055Zm-.43.852c.055.023.118.008.141-.047l.328-.648c.024-.047 0-.11-.047-.141-.054-.024-.117-.008-.14.047l-.328.648c-.024.047 0 .11.047.14ZM8.086 17.797c.054.039.07.117.039.172l-.906 1.351a.124.124 0 0 1-.172.032.135.135 0 0 1-.04-.172l.907-1.352a.125.125 0 0 1 .172-.031Zm-1.031 1.539c.046.031.117.023.148-.023l.906-1.352a.102.102 0 0 0-.031-.14c-.047-.032-.117-.024-.148.023l-.907 1.343a.111.111 0 0 0 .032.149ZM6.953 18.086c.055.039.063.117.023.172l-.453.57c-.039.055-.117.063-.171.016-.055-.04-.063-.117-.024-.172l.453-.57c.04-.055.117-.063.172-.016Zm-.586.742c.047.04.11.032.149-.015l.453-.57a.11.11 0 0 0-.024-.15c-.047-.038-.11-.03-.148.016l-.453.57c-.031.048-.024.118.023.15ZM7.031 16.914a.123.123 0 0 1 0 .172l-1.156 1.149a.122.122 0 0 1-.172 0 .122.122 0 0 1 0-.172l1.156-1.149a.123.123 0 0 1 .172 0Zm-1.32 1.313c.039.039.11.039.148 0l1.157-1.149a.108.108 0 0 0 0-.148.108.108 0 0 0-.149 0l-1.156 1.148c-.04.04-.04.102 0 .149ZM5.86 16.977c.046.054.038.132-.009.171l-.554.47c-.047.046-.125.038-.172-.016-.047-.055-.04-.133.008-.172l.554-.47c.047-.046.125-.038.172.017Zm-.72.617c.04.047.102.047.149.015l.555-.468a.109.109 0 0 0 .007-.149c-.039-.047-.101-.047-.148-.015l-.555.468c-.047.04-.047.102-.008.149ZM6.172 15.86a.125.125 0 0 1-.031.171l-1.36.907a.12.12 0 0 1-.171-.04.125.125 0 0 1 .03-.171L6 15.82a.12.12 0 0 1 .172.04Zm-1.547 1.023a.102.102 0 0 0 .14.031l1.36-.906c.047-.031.063-.094.024-.148a.102.102 0 0 0-.141-.032l-1.36.906a.11.11 0 0 0-.023.149ZM5.008 15.688c.031.062.016.132-.039.164l-.633.359c-.054.031-.133.008-.164-.047-.031-.063-.015-.133.04-.164l.632-.36c.055-.03.133-.007.164.047Zm-.828.468c.031.055.094.07.14.04l.633-.36a.112.112 0 0 0 .04-.14c-.032-.055-.094-.071-.141-.04l-.633.36c-.047.023-.063.086-.04.14ZM5.531 14.648a.13.13 0 0 1-.062.165l-1.508.624a.122.122 0 0 1-.156-.07.13.13 0 0 1 .062-.164l1.508-.625a.121.121 0 0 1 .156.07Zm-1.71.711a.111.111 0 0 0 .132.063l1.508-.625a.106.106 0 0 0 .055-.14.111.111 0 0 0-.133-.063l-1.508.625a.106.106 0 0 0-.055.14ZM4.43 14.258a.12.12 0 0 1-.079.156l-.687.227c-.063.023-.133-.016-.156-.078a.12.12 0 0 1 .078-.157l.687-.226a.133.133 0 0 1 .157.078Zm-.907.297c.016.055.079.086.133.07l.688-.226c.054-.016.078-.079.062-.133-.015-.055-.078-.086-.133-.07l-.687.226a.108.108 0 0 0-.063.133ZM5.14 13.336a.126.126 0 0 1-.093.148l-1.602.313c-.062.015-.125-.031-.14-.102a.125.125 0 0 1 .093-.148L5 13.234c.063-.015.125.032.14.102Zm-1.82.352a.105.105 0 0 0 .118.085l1.601-.312c.055-.008.094-.07.078-.125A.105.105 0 0 0 5 13.25l-1.602.313c-.054.015-.093.07-.078.125ZM4.133 12.734c.007.07-.04.133-.102.141l-.719.086a.121.121 0 0 1-.132-.11c-.008-.07.039-.132.101-.14L4 12.625a.122.122 0 0 1 .133.11Zm-.946.11c.008.054.063.101.118.094l.718-.086a.105.105 0 0 0 .086-.118c-.008-.054-.062-.101-.117-.093l-.719.086a.105.105 0 0 0-.086.117ZM5.008 11.977c0 .07-.055.125-.117.125H3.258c-.063 0-.117-.055-.117-.125s.054-.125.117-.125H4.89c.062 0 .117.054.117.125Zm-1.852-.008c0 .055.047.11.102.11H4.89c.054 0 .101-.048.101-.11 0-.055-.047-.11-.101-.11H3.258c-.055.008-.102.055-.102.11ZM4.14 11.188a.127.127 0 0 1-.125.117l-.726-.055c-.063-.008-.117-.063-.11-.133A.127.127 0 0 1 3.304 11l.727.055c.062.008.11.07.11.133Zm-.953-.07c-.008.054.04.109.094.109l.727.054c.054 0 .101-.039.109-.101a.099.099 0 0 0-.094-.11l-.726-.054c-.055 0-.102.046-.11.101ZM5.148 10.617a.127.127 0 0 1-.14.102l-1.602-.32a.126.126 0 0 1-.093-.149.127.127 0 0 1 .14-.102l1.602.32c.062.016.101.079.093.15Zm-1.82-.367a.112.112 0 0 0 .078.125l1.602.32a.11.11 0 0 0 .125-.086.112.112 0 0 0-.078-.125l-1.602-.32c-.062-.008-.11.031-.125.086ZM4.445 9.68c-.015.062-.086.11-.148.086l-.703-.196a.125.125 0 0 1-.086-.148c.015-.063.086-.11.148-.086l.703.195c.063.016.102.079.086.149Zm-.914-.258c-.015.055.016.117.07.133l.704.195c.054.016.11-.016.125-.078.015-.055-.016-.117-.07-.133l-.704-.195a.112.112 0 0 0-.125.078ZM5.547 9.312a.121.121 0 0 1-.156.07l-1.508-.624c-.063-.024-.086-.094-.063-.164a.121.121 0 0 1 .157-.07l1.508.624a.13.13 0 0 1 .062.164Zm-1.71-.71a.106.106 0 0 0 .054.14l1.508.625c.054.024.109-.008.132-.062a.106.106 0 0 0-.054-.14l-1.508-.626c-.055-.016-.117.008-.133.063ZM5.04 8.258a.12.12 0 0 1-.165.055l-.648-.329a.12.12 0 0 1-.055-.164.12.12 0 0 1 .164-.054l.648.328c.063.023.086.101.055.164Zm-.845-.43c-.023.055-.007.117.047.14l.649.329c.047.023.11 0 .14-.047.024-.055.008-.117-.046-.14l-.657-.329a.096.096 0 0 0-.133.047ZM6.188 8.11a.135.135 0 0 1-.172.039l-1.352-.907a.123.123 0 0 1-.031-.172.135.135 0 0 1 .172-.039l1.351.907c.063.039.07.117.032.171Zm-1.54-1.032c-.03.047-.023.117.024.149l1.36.906c.046.031.109.016.14-.031.031-.047.023-.118-.023-.149l-1.36-.906a.102.102 0 0 0-.14.031ZM5.898 6.977c-.039.054-.117.062-.171.023l-.57-.445c-.055-.04-.063-.117-.024-.172.039-.055.117-.063.172-.023l.57.445c.055.039.063.117.023.172Zm-.75-.586c-.039.047-.03.109.016.148l.57.445a.11.11 0 0 0 .149-.023c.039-.047.031-.11-.016-.148l-.57-.446a.11.11 0 0 0-.149.024ZM7.055 7.055a.123.123 0 0 1-.172 0L5.727 5.906a.123.123 0 0 1 0-.172.123.123 0 0 1 .171 0l1.157 1.149a.11.11 0 0 1 0 .172ZM5.742 5.75c-.039.039-.039.11 0 .148L6.9 7.047c.039.039.109.039.148 0a.108.108 0 0 0 0-.149L5.891 5.742c-.04-.039-.102-.039-.149.008ZM6.992 5.883a.123.123 0 0 1-.172-.008l-.476-.547c-.047-.047-.04-.125.015-.172a.122.122 0 0 1 .172.008l.477.547c.039.055.039.133-.016.172Zm-.625-.711c-.047.039-.047.101-.015.148l.476.547c.04.04.102.047.149.008.046-.039.046-.102.015-.148l-.476-.547c-.04-.047-.102-.047-.149-.008ZM8.11 6.188a.125.125 0 0 1-.172-.032l-.906-1.351a.12.12 0 0 1 .039-.172.125.125 0 0 1 .171.031l.907 1.352a.119.119 0 0 1-.04.172Zm-1.032-1.54a.102.102 0 0 0-.03.141l.905 1.352c.032.047.094.054.149.023a.102.102 0 0 0 .03-.14L7.228 4.68a.111.111 0 0 0-.149-.031ZM8.273 5.031c-.062.032-.132.016-.164-.039L7.75 4.36c-.031-.054-.016-.132.047-.164.062-.03.133-.015.164.04l.36.632c.03.055.015.125-.048.164Zm-.468-.828c-.047.031-.07.094-.04.14l.36.634a.112.112 0 0 0 .14.039c.048-.032.07-.094.04-.141l-.36-.633c-.023-.047-.093-.07-.14-.039ZM9.313 5.547a.13.13 0 0 1-.165-.063l-.632-1.507a.121.121 0 0 1 .07-.157.13.13 0 0 1 .164.063l.633 1.508c.023.054-.008.125-.07.156Zm-.711-1.71c-.055.022-.079.085-.055.132l.633 1.508c.023.054.086.078.14.054.055-.023.078-.086.055-.133l-.633-1.507c-.031-.055-.086-.078-.14-.055ZM9.703 4.438a.121.121 0 0 1-.156-.07L9.32 3.68c-.023-.063.016-.133.079-.156a.121.121 0 0 1 .156.07l.226.687a.133.133 0 0 1-.078.157Zm-.305-.899c-.054.016-.085.078-.07.133l.227.688c.015.054.078.078.133.062.054-.016.085-.078.07-.133l-.227-.687a.099.099 0 0 0-.133-.063ZM10.617 5.148a.126.126 0 0 1-.148-.093l-.328-1.602c-.016-.062.031-.133.094-.14a.126.126 0 0 1 .148.093l.328 1.602c.016.062-.023.125-.094.14Zm-.367-1.82c-.055.008-.094.063-.086.125l.328 1.602c.008.054.07.093.125.078.055-.008.094-.063.086-.125l-.328-1.602a.112.112 0 0 0-.125-.078ZM11.21 4.133c-.07.008-.132-.04-.14-.102l-.094-.719a.122.122 0 0 1 .11-.132c.07-.008.133.039.14.101L11.32 4a.122.122 0 0 1-.11.133Zm-.116-.938c-.055.008-.102.063-.094.117l.094.72a.105.105 0 0 0 .117.085c.054-.008.101-.062.094-.117l-.094-.719a.105.105 0 0 0-.117-.086Z"
        fill="#fff"
      />
      <path
        d="M12.055 21.242a9.25 9.25 0 1 0 0-18.5 9.25 9.25 0 0 0 0 18.5Z"
        fill={`url(#${id}1kqfjmj_c)`}
        fillOpacity={0.2}
      />
      <path
        d="M18.46 6.125 11 10.906h-.008v.008l-.008.008-4.656 7.68 6.844-5.508.008-.008v-.008l5.28-6.953Z"
        fill="#000"
        fillOpacity={0.05}
      />
      <path d="m18.32 5.68-7.375 5.242 2.188 2.172L18.32 5.68Z" fill="#CD151E" />
      <path d="m10.953 10.906 1.102 1.086L18.32 5.68l-7.367 5.226Z" fill="#FA5153" />
      <path d="m10.953 10.906 2.188 2.172-7.375 5.242 5.187-7.414Z" fill="#ACACAC" />
      <path d="m5.766 18.32 6.289-6.328-1.102-1.086-5.187 7.414Z" fill="#EEE" />
      <defs>
        <linearGradient
          id={`${id}1kqfjmj_a`}
          x1={12}
          y1={22}
          x2={12}
          y2={2}
          gradientUnits="userSpaceOnUse"
        >
          <stop offset={0.25} stopColor="#DBDBDA" />
          <stop offset={1} stopColor="#fff" />
        </linearGradient>
        <linearGradient
          id={`${id}1kqfjmj_c`}
          x1={11.879}
          y1={9.347}
          x2={7.562}
          y2={17.093}
          gradientUnits="userSpaceOnUse"
        >
          <stop stopOpacity={0} />
          <stop offset={1} />
        </linearGradient>
        <radialGradient
          id={`${id}1kqfjmj_b`}
          cx={0}
          cy={0}
          r={1}
          gradientUnits="userSpaceOnUse"
          gradientTransform="translate(13.3 9.968) scale(11.2909)"
        >
          <stop stopColor="#2ABCE1" />
          <stop offset={0.114} stopColor="#2ABBE1" />
          <stop offset={1} stopColor="#3375F8" />
        </radialGradient>
      </defs>
    </svg>
  );
});
