import { useMemo, useRef } from 'react';

import { ApolloError, QueryHookOptions, QueryResult } from '@apollo/client';

type UseQuerySpec<T> = (baseOptions?: QueryHookOptions<any, any>) => QueryResult<T>;
type MapperSpec<T, R> = (data: T) => R;

type LoadingResult = { data: null; error: null; isLoading: true };
type SuccessResult<T> = { data: T; error: null; isLoading: false };
type ErrorResult = { data: null; error: ApolloError; isLoading: false };

type MappedQueryResult<T> = LoadingResult | SuccessResult<T> | ErrorResult;

export type ExtractMappedQueryResult<T extends () => any> = ReturnType<T> extends MappedQueryResult<
  infer P
>
  ? P
  : never;

export function useMappedQuery<T, R = T>(
  hook: UseQuerySpec<T>,
  mapper?: MapperSpec<T, R>,
): MappedQueryResult<R> {
  const result = hook();
  const mapperRef = useRef<MapperSpec<T, R>>();

  mapperRef.current = mapper;

  const data = useMemo(() => {
    if (result.data === undefined || result.data === null) {
      return null;
    }

    if (mapperRef.current) {
      return mapperRef.current(result.data);
    }

    return result.data;
  }, [result.data]);

  return {
    data,
    error: result.error,
    isLoading: result.loading,
  } as MappedQueryResult<R>;
}
