import { combine, createDomain, sample } from 'effector';
import { persist } from 'effector-storage';

import { createCookieAdapter } from '@client/shared/libs/effector-helpers';

import { getRandomByWeight } from './libs/get-random-by-weight';
import type { Banner, BannerId, Group, GroupDisplay } from './types';

export type BannerSystemSpec = ReturnType<typeof createBannerSystem>;

export function createBannerSystem(name: string) {
  const domain = createDomain('banner-system');

  const set = domain.createEvent<Group>();
  const dismiss = domain.createEvent();

  const hide = domain.createEvent<BannerId | BannerId[]>();

  const $key = domain.createStore('');
  const $display = domain.createStore<GroupDisplay>('one');
  const $shownIds = domain.createStore<BannerId[]>([]);
  const $banners = domain.createStore<Banner[]>([]);

  const $banner = combine([$shownIds, $banners]).map<Banner | null>(([shown, banners]) => {
    const availableBanners = banners.filter((banner) => !shown.includes(banner.id));

    if (availableBanners.length > 0) {
      return getRandomByWeight(availableBanners);
    }

    return null;
  });

  const $value = $banner.map((banner) => (banner ? banner.value : null));

  $key.on(set, (_, group) => group.key);
  $display.on(set, (_, group) => group.display);
  $banners.on(set, (_, group) => group.banners);

  $shownIds.on(hide, (ids, id) => (Array.isArray(id) ? [...ids, ...id] : [...ids, id]));

  set.watch((payload) => {
    persist({
      store: $shownIds,
      keyPrefix: name,
      key: payload.key,
      adapter: createCookieAdapter({ defaultValue: [], expires: 1 }),
    });
  });

  sample({
    clock: sample({
      clock: dismiss,
      source: $display,
      filter: (display) => display === 'all',
    }),
    source: $banner,
    filter: (banner) => banner !== null,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    fn: (banner) => banner!.id,
    target: hide,
  });

  sample({
    clock: sample({
      clock: dismiss,
      source: $display,
      filter: (display) => display === 'one',
    }),
    source: $banners,
    fn: (banners) => banners.map((banner) => banner.id),
    target: hide,
  });

  return {
    $value,
    dismiss,
    set,
  };
}
