import { Event, Store, combine, guard, sample } from 'effector';

import { createFormUnit } from './create-form-unit';
import { bindChangeEvent, bindValidation, createField } from './field';
import { AnyFormValues, Fields, FormConfig } from './types';
import { eachValid } from './validation';

function createFormValuesStore<T extends AnyFormValues>(fields: Fields<T>) {
  const shape = {} as { [K in keyof T]: Store<T[K]> };

  for (const fieldName in fields) {
    if (!fields.hasOwnProperty(fieldName)) {
      continue;
    }

    shape[fieldName] = fields[fieldName].$value;
  }

  return combine(shape) as Store<T>;
}

export type Form<FormValues extends AnyFormValues> = {
  fields: Fields<FormValues>;
  $values: Store<FormValues>;
  $eachValid: Store<boolean>;
  $isValid: Store<boolean>;
  $isDirty: Store<boolean>;
  $isTouched: Store<boolean>;
  $touched: Store<boolean>;
  $meta: Store<{
    isValid: boolean;
    isDirty: boolean;
    isTouched: boolean;
  }>;
  submit: Event<void>;
  validate: Event<void>;
  reset: Event<void>;
  set: Event<Partial<FormValues>>;
  setForm: Event<Partial<FormValues>>;
  resetTouched: Event<void>;
  resetValues: Event<void>;
  resetErrors: Event<void>;
  formValidated: Event<FormValues>;
};

export function createForm<FormValues extends AnyFormValues>(
  config: FormConfig<FormValues>,
): Form<FormValues> {
  const { filter: $filter, domain, fields: fieldsConfigs, validateOn, units } = config;

  const fields = {} as Fields<FormValues>;

  const dirtyFlagsArr: Store<boolean>[] = [];
  const touchedFlagsArr: Store<boolean>[] = [];

  // create units
  for (const fieldName in fieldsConfigs) {
    if (!fieldsConfigs.hasOwnProperty(fieldName)) {
      continue;
    }

    const fieldConfig = fieldsConfigs[fieldName];

    const field = createField(fieldName, fieldConfig, domain);

    fields[fieldName] = field;

    dirtyFlagsArr.push(field.$isDirty);
    touchedFlagsArr.push(field.$isTouched);
  }

  const $form = createFormValuesStore(fields);
  const $eachValid = eachValid(fields);
  const $isFormValid = $filter
    ? combine($eachValid, $filter, (valid, filter) => valid && filter)
    : $eachValid;
  const $isDirty = combine(dirtyFlagsArr).map((dirtyFlags) => dirtyFlags.some(Boolean));
  const $isTouched = combine(touchedFlagsArr).map((touchedFlags) => touchedFlags.some(Boolean));

  const $meta = combine({
    isValid: $eachValid,
    isDirty: $isDirty,
    isTouched: $isTouched,
  });

  // define events
  const validate = createFormUnit.event({ domain, existing: units?.validate });
  const submitForm = createFormUnit.event({ domain, existing: units?.submit });
  const formValidated = createFormUnit.event({ domain, existing: units?.formValidated });
  const setForm = createFormUnit.event({ domain, existing: units?.setForm });
  const resetForm = createFormUnit.event({ domain, existing: units?.reset });
  const resetValues = createFormUnit.event({ domain, existing: units?.resetValues });
  const resetErrors = createFormUnit.event({ domain, existing: units?.resetErrors });
  const resetTouched = createFormUnit.event({ domain, existing: units?.resetTouched });
  const submitWithFormData = sample({ source: $form, clock: submitForm }) as Event<FormValues>;
  const validateWithFormData = sample({ source: $form, clock: validate }) as Event<FormValues>;

  // bind units
  for (const fieldName in fields) {
    if (!fields.hasOwnProperty(fieldName)) {
      continue;
    }

    const fieldConfig = fieldsConfigs[fieldName];
    const field = fields[fieldName];

    bindChangeEvent(field, setForm, resetForm, resetTouched, resetValues);

    if (!fieldConfig.rules) {
      continue;
    }

    bindValidation({
      $form,
      rules: fieldConfig.rules,
      submitEvent: submitForm,
      resetFormEvent: resetForm,
      resetValues,
      resetErrors,
      validateFormEvent: validate,
      field,
      formValidationEvents: validateOn ? validateOn : ['submit'],
      fieldValidationEvents: fieldConfig.validateOn ? fieldConfig.validateOn : [],
    });
  }

  guard({
    source: submitWithFormData,
    filter: $isFormValid,
    target: formValidated,
  });

  guard({
    source: validateWithFormData,
    filter: $isFormValid,
    target: formValidated,
  });

  return {
    fields,
    $values: $form,
    $eachValid,
    $isValid: $eachValid,
    $isDirty,
    $isTouched,
    $touched: $isTouched,
    $meta,
    submit: submitForm,
    validate,
    resetTouched,
    reset: resetForm,
    resetValues,
    resetErrors,
    setForm,
    set: setForm,
    formValidated,
  };
}
