import { Event } from 'effector';
import { useStore } from 'effector-react';
import { useEffect, useRef, useState } from 'react';

import { Form } from './factory';
import { AnyFormValues, Field, ValidationError } from './types';

type ErrorTextMap = {
  [key: string]: string;
};

type ConnectedField<Value> = {
  name: string;
  value: Value;
  errors: ValidationError<Value>[];
  firstError: ValidationError<Value> | null;
  hasError: () => boolean;
  onChange: Event<Value>;
  onBlur: Event<void>;
  errorText: (map?: ErrorTextMap) => string | undefined;
  addError: Event<{ rule: string; errorText?: string }>;
  validate: Event<void>;
  isValid: boolean;
  isDirty: boolean;
  isTouched: boolean;
  reset: Event<void>;
  set: Event<Value>;
  resetErrors: Event<void>;
};

type ConnectedFields<FormValues extends AnyFormValues = AnyFormValues> = {
  [K in keyof FormValues]: ConnectedField<FormValues[K]>;
};

export function useField<Value>(field: Field<Value>): ConnectedField<Value> {
  const { value, errors, firstError, isValid, isDirty, isTouched } = useStore(field.$field);

  return {
    name: field.name,
    value,
    errors,
    firstError,
    isValid,
    isDirty,
    isTouched,
    onChange: field.onChange,
    onBlur: field.onBlur,
    addError: field.addError,
    validate: field.validate,
    reset: field.reset,
    set: field.onChange,
    resetErrors: field.resetErrors,
    hasError: () => {
      return firstError !== null;
    },
    errorText: (map) => {
      if (!firstError) {
        return undefined;
      }

      if (!map) {
        return firstError.errorText;
      }

      if (map[firstError.rule]) {
        return map[firstError.rule];
      }

      return firstError.errorText;
    },
  };
}

export type FormState<FormValues extends AnyFormValues = AnyFormValues> = {
  fields: ConnectedFields<FormValues>;
  values: FormValues;
  hasError: (fieldName?: keyof FormValues) => boolean;
  addError: (fieldName: keyof FormValues, error: { rule: string; errorText?: string }) => void;
  eachValid: boolean;
  isValid: boolean;
  isDirty: boolean;
  isTouched: boolean;
  errors: (fieldName: keyof FormValues) => ValidationError<FormValues[typeof fieldName]>[];
  error: (fieldName: keyof FormValues) => ValidationError<FormValues[typeof fieldName]> | null;
  errorText: (fieldName: keyof FormValues, map?: ErrorTextMap) => string;
  submit: Event<void>;
  reset: Event<void>;
  setForm: Event<Partial<FormValues>>;
  set: Event<Partial<FormValues>>;
  formValidated: Event<FormValues>;
};

export function useForm<FormValues extends AnyFormValues>(
  formInit: Form<FormValues> | (() => Form<FormValues>),
) {
  const [form] = useState(formInit);
  const connectedFields = {} as ConnectedFields;
  const values = {} as AnyFormValues;

  for (const fieldName in form.fields) {
    if (!form.fields.hasOwnProperty(fieldName)) continue;
    const field = form.fields[fieldName];
    // eslint-disable-next-line react-hooks/rules-of-hooks
    const connectedField = useField(field);

    connectedFields[fieldName] = connectedField;
    values[fieldName] = connectedField.value;
  }

  const { isValid: eachValid, isDirty, isTouched } = useStore(form.$meta);

  const hasError = (fieldName?: string): boolean => {
    if (!fieldName) {
      return !eachValid;
    }
    if (connectedFields[fieldName]) {
      return Boolean(connectedFields[fieldName].firstError);
    }

    return false;
  };

  const error = (fieldName: string) => {
    if (connectedFields[fieldName]) {
      return connectedFields[fieldName].firstError;
    }

    return null;
  };

  const errors = (fieldName: string) => {
    if (connectedFields[fieldName]) {
      return connectedFields[fieldName].errors;
    }

    return [];
  };

  const errorText = (fieldName: string, map?: ErrorTextMap) => {
    const field = connectedFields[fieldName];

    if (!field) {
      return '';
    }

    if (!field.firstError) {
      return '';
    }

    if (!map) {
      return field.firstError.errorText || '';
    }

    if (map[field.firstError.rule]) {
      return map[field.firstError.rule];
    }

    return field.firstError.errorText || '';
  };

  return {
    fields: connectedFields,
    values,
    hasError,
    eachValid,
    isValid: eachValid,
    isDirty,
    isTouched,
    errors,
    error,
    reset: form.reset,
    errorText,
    submit: form.submit,
    setForm: form.setForm,
    set: form.setForm, // set form alias
    formValidated: form.formValidated,
  } as FormState<FormValues>;
}

export interface UseFormEventsProps<T> {
  onSubmit?: (payload: T) => void;
}

export function useFormEvents<T>(form: FormState<T>, props: UseFormEventsProps<T>) {
  const propsRef = useRef(props);
  const { formValidated } = form;

  useEffect(() => {
    return formValidated.watch((payload) => {
      propsRef.current.onSubmit?.(payload);
    });
  }, [formValidated]);
}
