import { Domain, Event, Store } from 'effector';

type InitFieldValue<Value> = () => Value;

export type ValidationEvent = 'submit' | 'blur' | 'change';

export type ValidationResult = {
  isValid: boolean;
  errorText?: string;
};

export type Validator<Value, Form = any, Source = any> = (
  value: Value,
  form: Form,
  source?: Source,
) => boolean | ValidationResult;

export interface ValidationErrorPayload {
  rule: string;
  errorText?: string;
}

export interface ValidationError<Value = any> extends ValidationErrorPayload {
  value: Value;
}

export type Rule<Value, Form = any, Source = any> = {
  name: string;
  errorText?: string;
  source?: Store<Source>;
  validator: Validator<Value, Form, Source>;
};

export type FieldData<Value> = {
  value: Value;
  errors: ValidationError<Value>[];
  firstError: ValidationError<Value> | null;
  isValid: boolean;
  isDirty: boolean;
  isTouched: boolean;
};

export type Field<Value> = {
  name: string;
  $value: Store<Value>;
  $errors: Store<ValidationError<Value>[]>;
  $firstError: Store<ValidationError<Value> | null>;
  $isValid: Store<boolean>;
  $isDirty: Store<boolean>;
  $isTouched: Store<boolean>;
  $field: Store<FieldData<Value>>;
  onChange: Event<Value>;
  changed: Event<Value>;
  onBlur: Event<void>;
  addError: Event<ValidationErrorPayload>;
  validate: Event<void>;
  reset: Event<void>;
  set: Event<Value>;
  resetErrors: Event<void>;
  resetValue: Event<void>;
  filter?: Store<boolean> | FilterFunc<Value>;
};

type FilterFunc<Value> = (value: Value) => boolean;

export type RuleResolver<Value = any, Form = any> = (
  value: Value,
  form: Form,
) => Rule<Value, Form, void>[];

export type FieldConfig<Value, FormValues extends AnyFormValues = AnyFormValues> = {
  init: Value | InitFieldValue<Value>;
  rules?: Rule<Value, FormValues>[] | RuleResolver<Value, FormValues>;
  filter?: Store<boolean> | FilterFunc<Value>;
  validateOn?: ValidationEvent[];
  units?: {
    $value?: Store<Value>;
    $errors?: Store<ValidationError<Value>[]>;
    $isTouched?: Store<boolean>;
    onChange?: Event<Value>;
    changed?: Event<Value>;
    onBlur?: Event<void>;
    addError?: Event<ValidationErrorPayload>;
    validate?: Event<void>;
    resetValue?: Event<void>;
    reset?: Event<void>;
    resetErrors?: Event<void>;
  };
};

export type AnyFormValues = {
  [key: string]: any;
};

export type Fields<FormValues extends AnyFormValues = AnyFormValues> = {
  [K in keyof FormValues]: Field<FormValues[K]>;
};

export type FieldsConfig<FormValues extends AnyFormValues> = {
  [K in keyof FormValues]: FieldConfig<FormValues[K], FormValues>;
};

export type FormConfig<FormValues extends AnyFormValues> = {
  fields: FieldsConfig<FormValues>;
  domain?: Domain;
  filter?: Store<boolean>;
  validateOn?: ValidationEvent[];
  units?: {
    submit?: Event<void>;
    validate?: Event<void>;
    reset?: Event<void>;
    resetValues?: Event<void>;
    resetTouched?: Event<void>;
    resetErrors?: Event<void>;
    formValidated?: Event<FormValues>;
    setForm?: Event<Partial<FormValues>>;
  };
};
