import { createEvent, createStore, restore, sample } from 'effector';
import Router from 'next/router';

const setDialog = createEvent<string | null>();

interface DialogPayload {
  key: string;
  params?: any;
}

export const hideDialog = createEvent<DialogPayload>();
export const showDialog = createEvent<DialogPayload>();

export const $dialog = restore<string | null>(setDialog, null);

showDialog.watch((payload) => {
  const { key, params } = payload;
  const query = { ...Router.query, ...params, dialog: key };

  Router.push({ pathname: Router.pathname, query }, undefined, { shallow: true, scroll: false });
});

hideDialog.watch((payload) => {
  const { key, params } = payload;
  const { dialog, ...query } = Router.query;

  if (dialog === key) {
    for (const param in params) {
      delete query[param];
    }

    Router.push({ pathname: Router.pathname, query }, undefined, { shallow: true, scroll: false });
  }
});

function sync() {
  const { dialog } = Router.query;

  setDialog((dialog as string) || null);
}

Router.ready(sync);
Router.events.on('routeChangeComplete', sync);

export function createDialogApi<T, U extends Record<string, any> = {}>(key: string | null = null) {
  const set = createEvent<T>();
  const setParams = createEvent<U>();
  const reset = createEvent();

  const show = createEvent<T>();
  const hide = createEvent();

  const $isVisible = createStore(false);
  const $state = createStore<T>({} as T);
  const $params = createStore<U>({} as U);

  $state.on(set, (_, payload) => payload);
  $state.on(show, (_, payload) => payload);
  $state.reset(reset);

  $params.on(setParams, (_, payload) => payload);
  $params.reset(reset);

  if (key === null) {
    $isVisible.on(show, () => true);
    $isVisible.on(hide, () => false);
  } else {
    sample({
      clock: show,
      source: $params,
      target: showDialog.prepend((params) => ({ key, params })),
    });
    sample({
      clock: hide,
      source: $params,
      target: hideDialog.prepend((params) => ({ key, params })),
    });
    sample({ clock: $dialog.map((dialog) => dialog === key), target: $isVisible });
  }

  return {
    $isVisible,
    $state,
    hide,
    key,
    reset,
    set,
    setParams,
    show,
  };
}
