import { canUseDom } from '@client/shared/libs/can-use-dom';

import { HydratableStore, Store, Type } from './types';

const $isHydrated = Symbol('mobx-next (isHydrated)');

type InternalStore = Partial<HydratableStore> & {
  [$isHydrated]?: boolean;
};

export class StoreManager {
  private static instance: StoreManager | null;

  static getInstance() {
    if (!this.instance) {
      const instance = new StoreManager();

      if (!canUseDom) {
        return instance;
      }

      this.instance = instance;
    }

    return this.instance;
  }

  private stores: Map<Type<InternalStore>, InternalStore> = new Map();

  get<T extends Store>(store: Type<T>, scope: unknown): T;
  get<T extends HydratableStore>(store: Type<T>, scope: unknown): T;
  get<T extends InternalStore>(Store: Type<T>, scope: unknown): T {
    const store = this.getOrCreateStore(Store);

    this.hydrate(store, scope);

    return store;
  }

  private getOrCreateStore<T extends InternalStore>(Store: Type<T>): T {
    let store = this.stores.get(Store);

    if (!store) {
      store = new Store();
      this.stores.set(Store, store);
    }

    return store as T;
  }

  private hydrate(store: InternalStore, scope: unknown) {
    if (!store[$isHydrated] && store.hydrate) {
      store.hydrate(scope);
      store[$isHydrated] = true;
    }
  }
}
