import { createEffect, createEvent, createStore, forward, sample } from 'effector';
import { ComponentType } from 'react';

import { ErrorSnackbar, LoadingSnackbar, SuccessSnackbar } from '@client/shared/ui-kit';
import { SnackbarComponentProps, SnackbarState, SnackbarStore } from '@yandex-id/components';

export const snackbarStore = new SnackbarStore();

interface VisibilityChangeEvent {
  id: string;
  visibility: 'pending' | 'visible' | 'hidden';
}

type VisibilityChangeHandler = (event: VisibilityChangeEvent) => void;

interface NotifierShowOptions {
  component: ComponentType<SnackbarComponentProps>;
  onVisibilityChange?: VisibilityChangeHandler;
}

interface NotifierUpdateOptions {
  id: string;
  component: ComponentType<SnackbarComponentProps>;
}

function createGlobalNotifier(store: SnackbarStore) {
  const visibilityChangeHandlers = new Map<string, VisibilityChangeHandler>();

  const $notifications = createStore<SnackbarState[]>(store.getQueue());

  const setNotifications = createEvent<SnackbarState[]>();
  const show = createEvent<NotifierShowOptions>();
  const hide = createEvent<string>();
  const update = createEvent<NotifierUpdateOptions>();
  const clear = createEvent();

  const showFx = createEffect((payload: NotifierShowOptions) => {
    const id = store.show(payload.component);

    if (payload.onVisibilityChange) {
      payload.onVisibilityChange({ id, visibility: 'pending' });

      visibilityChangeHandlers.set(id, payload.onVisibilityChange);
    }

    return id;
  });

  const hideFx = createEffect((id: string) => {
    store.hide(id);

    return id;
  });

  const updateFx = createEffect((payload: NotifierUpdateOptions) => {
    return store.update(payload.id, payload.component);
  });
  const clearFx = createEffect(() => store.clear());

  forward({ from: show, to: showFx });
  forward({ from: hide, to: hideFx });
  forward({ from: update, to: updateFx });
  forward({ from: clear, to: clearFx });

  $notifications.on(setNotifications, (_, ids) => ids);

  store.subscribe(() => {
    const current = $notifications.getState();
    const next = store.getQueue();

    const currentIds = current.map((item) => item.id);
    const nextIds = next.map((item) => item.id);

    const hiddenIds = currentIds.filter((id) => !nextIds.includes(id));

    hiddenIds.forEach((id) => {
      const onVisibilityChange = visibilityChangeHandlers.get(id);

      if (onVisibilityChange) {
        visibilityChangeHandlers.delete(id);

        onVisibilityChange({ id, visibility: 'hidden' });
      }
    });

    const currentVisibleIds = current
      .filter((item) => item.visible && item.visibility === 'visible')
      .map((item) => item.id);
    const nextVisibleIds = next
      .filter((item) => item.visible && item.visibility === 'visible')
      .map((item) => item.id);

    const newVisibleIds = nextVisibleIds.filter((id) => !currentVisibleIds.includes(id));

    newVisibleIds.forEach((id) => {
      const onVisibilityChange = visibilityChangeHandlers.get(id);

      if (onVisibilityChange) {
        onVisibilityChange({ id, visibility: 'visible' });
      }
    });

    setNotifications(next);
  });

  return {
    show,
    hide,
    clear,
  };
}

const globalNotifier = createGlobalNotifier(snackbarStore);

export function createNotifier() {
  const $ids = createStore<string[]>([]);

  const updateId = createEvent<VisibilityChangeEvent>();

  const show = globalNotifier.show.prepend((payload: NotifierShowOptions): NotifierShowOptions => {
    const { onVisibilityChange } = payload;

    return {
      ...payload,
      onVisibilityChange: (event: VisibilityChangeEvent) => {
        updateId(event);

        onVisibilityChange?.(event);
      },
    };
  });
  const hide = createEvent();

  const hideFx = createEffect((ids: string[]) => {
    ids.forEach((id) => {
      globalNotifier.hide(id);
    });
  });

  $ids.on(updateId, (state, payload) => {
    const { id, visibility } = payload;

    if (visibility === 'hidden' && state.includes(id)) {
      return state.filter((id) => id !== payload.id);
    }

    if (visibility !== 'hidden' && !state.includes(id)) {
      return state.concat(id);
    }

    return state;
  });

  sample({
    clock: [show, hide],
    source: $ids,
    filter: $ids.map((ids) => ids.length > 0),
    target: hideFx,
  });

  return {
    show,
    hide,
    success: show.prepend((text: string) => {
      return {
        component: (props) => {
          return <SuccessSnackbar {...props}>{text}</SuccessSnackbar>;
        },
      };
    }),
    error: show.prepend((text: string) => {
      return {
        component: (props) => {
          return <ErrorSnackbar {...props}>{text}</ErrorSnackbar>;
        },
      };
    }),
    loading: show.prepend((text: string) => {
      return {
        component: (props) => {
          return <LoadingSnackbar {...props}>{text}</LoadingSnackbar>;
        },
      };
    }),
  };
}
