import fs from 'fs';

import serialize from 'serialize-javascript';

import { LoggerService } from '@server/shared/logger';
import { ServerSideProps } from '@shared/types/next-bridge';

const emptyInlineRumBundle = [
  '(window.Ya || (window.Ya = {})).Rum = {',
  'logError: () => {},',
  'ERROR_LEVEL: {},',
  'sendTimeMark: () => {},',
  'time: () => {},',
  'timeEnd: () => {},',
  'getTime: () => {},',
  'sendDelta: () => {},',
  'updateAdditional: () => {}',
  '};',
].join('');

export class RumInline {
  private inlineBundlePrefix = emptyInlineRumBundle;
  constructor(filePath: string, private logger: LoggerService) {
    this.loadInlineRumBundle(filePath);
  }

  getInlineBundle(props: ServerSideProps, locale?: string) {
    return [
      '(window.Ya || (window.Ya = {})).id =',
      serialize(
        {
          uid: props?.viewer?.uid,
          reqid: props?.reqid,
          version: props?.version,
          locale: locale,
          yenv: props?.yenv,
          isTouch: props?.ua?.isTouch,
        },
        {
          isJSON: true,
        },
      ),
      ';',
      this.inlineBundlePrefix,
    ].join('');
  }

  getInlineScript(props: ServerSideProps, nonce?: string, locale?: string) {
    const inlineRumBundle = {
      __html: this.getInlineBundle(props, locale),
    };

    return <script nonce={nonce} dangerouslySetInnerHTML={inlineRumBundle} />;
  }

  private loadInlineRumBundle(filePath: string) {
    try {
      this.inlineBundlePrefix = fs.readFileSync(filePath, 'utf8');
    } catch (err) {
      this.logger.error('Failed to load inline rum bundle', { err });
    }
  }
}
