import { Duration, intervalToDuration } from 'date-fns';
import { useEffect, useMemo, useRef, useState } from 'react';

export interface UseTimerResult {
  milliseconds: number;
  duration: Duration;
}

export interface UseTimerProps {
  interval?: number;
  expiryTimestamp?: number;
  onExpire?: () => void;
}

export function useTimer(props: UseTimerProps): UseTimerResult {
  const { expiryTimestamp: expiry, onExpire, interval = 1000 } = props;
  const refExpire = useRef(onExpire);
  const [timestamp, setTimestamp] = useState<number | null>(null);
  const expiryTimestamp = expiry ? floorMilliseconds(expiry) : undefined;

  refExpire.current = onExpire;

  useEffect(() => {
    const initialTimestamp = getCurrentTimestamp();

    setTimestamp(initialTimestamp);

    if (!expiryTimestamp || initialTimestamp >= expiryTimestamp) {
      return;
    }

    let timer: number | undefined;
    const timerCb = () => {
      const now = getCurrentTimestamp();

      setTimestamp(now);

      if (now < expiryTimestamp) {
        timer = window.setTimeout(timerCb, interval);
      } else {
        refExpire.current?.();
      }
    };

    timerCb();

    return () => {
      window.clearTimeout(timer);
    };
  }, [interval, expiryTimestamp]);

  const milliseconds =
    !expiryTimestamp || !timestamp ? 0 : Math.max(expiryTimestamp - timestamp, 0);

  const duration = useMemo(() => {
    return intervalToDuration({ start: 0, end: milliseconds });
  }, [milliseconds]);

  return { milliseconds, duration };
}

function floorMilliseconds(value: number) {
  return Math.floor(value / 1000) * 1000;
}

function getCurrentTimestamp() {
  return floorMilliseconds(Number(new Date()));
}
