import cx from 'classnames';
import React, {
  ForwardedRef,
  ReactNode,
  forwardRef,
  useCallback,
  useEffect,
  useMemo,
  useRef,
  useState,
} from 'react';

import { useClientHeight } from '@client/shared/libs/use-client-height';
import { useKeyBinding, usePress } from '@use-platform/react';
import { Portal, PortalExtendableProps } from '@yandex-lego/components/Portal';
import { usePreventScroll } from '@yandex-lego/components/usePreventScroll';

import { getSpringTransform, noop, useDrag, useSpring } from './utils';

import styles from './Drawer.module.css';

export interface DrawerAnimationParams {
  /**
   * Основные параметры анимации для rebound
   * @see https://github.com/facebook/rebound-js
   *
   * @default 230
   */
  tension: number;
  /**
   * @default 24
   */
  friction: number;
  /**
   * Отключает спринговые анимации (напр. для прогонов автотестов)
   */
  disabled?: boolean;
  /**
   * Отключает анимации на момент перетаскивания шторки пальцем (таким образом делая их отзывчивее)
   */
  dragImmediate?: boolean;
}

interface DragStateData {
  isTargetUnderContent: boolean;
  initialScrollPosition: number;
  isScrolled: boolean;
}

export interface DrawerProps extends PortalExtendableProps {
  zIndex?: number;
  onClose?: () => void;
  onCloseEnd?: () => void;
  direction?: 'bottom' | 'left' | 'right';
  maxSize?: string;
  animation?: DrawerAnimationParams;
  className?: string;
  children: ReactNode;
}

const defaultAnimation: DrawerAnimationParams = {
  dragImmediate: false,
  disabled: false,
  tension: 230,
  friction: 24,
};

function _Drawer(props: DrawerProps, ref: ForwardedRef<HTMLDivElement>) {
  const {
    className,
    visible,
    direction = 'bottom',
    animation = defaultAnimation,
    children,
    onClose = noop,
    onCloseEnd = noop,
    maxSize,
    scope,
    keepMounted,
    zIndex,
  } = props;

  // прогресс открытия шторки от 0 до 1
  const [progress, setProgress] = useState(0);

  // признак того, что анимация временно отключена (напр. на время drag жеста)
  const [springDisabled, setSpringDisabled] = useState(false);

  const contentRef = useRef<HTMLDivElement>(null);
  const [closing, setClosing] = useState(false);

  // спринговое значение прогресса и его производные
  const springImmediate = (animation.dragImmediate && springDisabled) || animation.disabled;
  const springValue = useSpring(progress, animation.tension, animation.friction, springImmediate);
  const springVisible = springValue > 0;
  const isDragged = progress < 1;

  // решает баг в iOS: в альбомной ориентации fixed элементы с
  // height: 100% показываются некорректно если виден navigation bar
  const clientHeight = useClientHeight();
  const popupStyle = { height: clientHeight && `${clientHeight}px`, zIndex };

  const _onClose = useCallback(() => {
    setClosing(true);
    onClose();
  }, [onClose]);

  usePreventScroll({ enabled: visible });

  useKeyBinding({ bind: 'Escape', onAction: _onClose });

  const { pressProps } = usePress({
    onPress: () => {
      if (!isDragged) {
        _onClose();
      }
    },
  });

  useEffect(() => {
    if (closing && springValue === 0) {
      onCloseEnd();
      setClosing(false);
    }
  }, [closing, springValue, onCloseEnd]);

  useEffect(() => {
    setSpringDisabled(false);
    setProgress(visible ? 1 : 0);
  }, [visible, animation]);

  const axis = direction === 'bottom' ? 'y' : 'x';
  const inverted = direction === 'left' ? -1 : 1;

  const springOpacity = Math.max(springValue, 0);
  const springTransform = getSpringTransform({ axis, springValue, inverted });

  const curtainStyle = useMemo(
    () => ({
      ...(springTransform ? { transform: springTransform } : {}),
      ...(maxSize && { [axis === 'x' ? 'maxWidth' : 'maxHeight']: maxSize }),
    }),
    [springTransform, maxSize, axis],
  );

  const { dragProps } = useDrag<DragStateData>((dragState) => {
    if (!visible || !contentRef.current) return;

    const {
      velocity: { x: vx, y: vy },
      movement: { x: mx, y: my },
      first,
      last,
      data,
      event,
    } = dragState;

    const drawerSize =
      axis === 'x' ? contentRef.current.clientWidth : contentRef.current.clientHeight;
    const movement = inverted * (axis === 'x' ? mx : my);
    const velocity = inverted * (axis === 'x' ? vx : vy);

    if (first) {
      data.isTargetUnderContent = contentRef.current.contains(event.target as HTMLElement);
      data.initialScrollPosition = contentRef.current.scrollTop;
      data.isScrolled = data.initialScrollPosition !== 0;
    } else {
      data.isScrolled =
        data.isScrolled || data.initialScrollPosition - contentRef.current.scrollTop < 0;
    }

    // ничего не делаем когда жест происходит одновременно с проскроллом
    // или если шторка в статичном состоянии
    // @see SERP-107544
    if (data.isTargetUnderContent && data.isScrolled) {
      return;
    }

    // жест завершен, возвращаем шторку в открытое положение, если
    // скорость была недостаточной и, закрываем если наоборот
    if (last) {
      setSpringDisabled(false);

      if (Math.abs(velocity) >= 0.1) {
        return velocity > 0 ? _onClose() : setProgress(1);
      } else if (movement / drawerSize >= 0.3) {
        return _onClose();
      }

      return setProgress(1);
    }

    if (movement > 0) {
      setSpringDisabled(true);

      const progress = Math.max(0, 1 - movement / drawerSize);

      if (progress === 0) {
        return _onClose();
      }

      return setProgress(progress);
    }
  });

  return (
    <Portal visible={springVisible} scope={scope} keepMounted={keepMounted}>
      <div
        className={cx(
          styles.root,
          {
            [styles.root_isVisible]: springVisible,
          },
          className,
        )}
        style={popupStyle}
        ref={ref}
      >
        <div className={styles.observer} {...dragProps}>
          <div className={styles.overlay} style={{ opacity: springOpacity }} {...pressProps} />
          <div className={styles.curtain} style={curtainStyle}>
            <div className={styles.thumb} />
            <div className={styles.content} ref={contentRef} data-slot="content">
              {children}
            </div>
          </div>
        </div>
      </div>
    </Portal>
  );
}

export const Drawer = forwardRef(_Drawer);
