import React, { FC, ImgHTMLAttributes, useCallback, useEffect, useMemo, useRef } from 'react';

import { ORD } from './CropImage.constants';
import { CropOrd, ImageInfo, Point, Rect } from './CropImage.types';
import {
  clamp,
  cn,
  getImageInfo,
  getInitialPoint,
  getInitialRect,
  getRelativeClientPos,
  initMouseMoveEvents,
  makeDefaultCrop,
  moveRect,
  setCropSelection,
} from './CropImage.utils';

import styles from './CropImage.module.css';

export interface CropImageProps extends ImgHTMLAttributes<HTMLImageElement> {
  crop?: Rect;
  minWidth?: number;
  minHeight?: number;
  maxWidth?: number;
  maxHeight?: number;
  aspectRatio?: number;
  guides?: boolean;
  circularShape?: boolean;
  disabled?: boolean;
  keepSelection?: boolean;
  locked?: boolean;
  onCropComplete?: (crop: Rect) => void;
  onCropChange?: (crop: Rect) => void;
}

export const CropImage: FC<CropImageProps> = (props) => {
  const {
    src,
    crop,
    minWidth,
    minHeight,
    maxWidth,
    maxHeight,
    aspectRatio,
    guides,
    circularShape,
    disabled,
    locked,
    keepSelection,
    className,
    children,
    onCropChange,
    onCropComplete,
    onLoad,
    ...other
  } = props;
  const containerElRef = useRef<HTMLDivElement>(null);
  const mediaElRef = useRef<HTMLImageElement>(null);
  const cropSelectRef = useRef<HTMLDivElement>(null);
  const cropRef = useRef<Rect>(getInitialRect());
  const imageInfoRef = useRef<ImageInfo>({
    size: { width: 0, height: 0 },
    offset: getInitialPoint(),
    scale: 1,
  });

  const cropOptions = useMemo(() => {
    return {
      aspect: aspectRatio,
      minHeight,
      minWidth,
      maxHeight,
      maxWidth,
    };
  }, [minHeight, minWidth, maxHeight, maxWidth, aspectRatio]);

  const handleImageLoad = useCallback(
    (e: React.SyntheticEvent<HTMLImageElement>) => {
      const containerEl = containerElRef.current;
      const mediaEl = mediaElRef.current;
      const cropSelectEl = cropSelectRef.current;

      if (mediaEl && containerEl && cropSelectEl) {
        imageInfoRef.current = getImageInfo(containerEl, mediaEl);
        const imageInfo = imageInfoRef.current;
        const nextCrop = makeDefaultCrop(imageInfo, cropOptions);

        cropRef.current = nextCrop;

        setCropSelection(cropSelectEl, imageInfo, nextCrop);

        if (crop !== nextCrop && onCropComplete) {
          onCropComplete(nextCrop);
        }
      }

      if (onLoad) {
        onLoad(e);
      }
    },
    [crop, cropOptions, onCropComplete, onLoad],
  );

  const handleContainerMouseDown = useCallback(
    (event: React.MouseEvent<HTMLDivElement> | React.TouchEvent<HTMLDivElement>) => {
      const containerEl = containerElRef.current;
      const cropEl = cropSelectRef.current;
      const mediaEl = mediaElRef.current;

      if (disabled || locked || keepSelection || !mediaEl) {
        return;
      }

      if (event.target !== mediaElRef.current || !containerEl || !cropEl) {
        return;
      }

      event.preventDefault();

      imageInfoRef.current = getImageInfo(containerEl, mediaEl);
      const imageInfo = imageInfoRef.current;
      const { size } = imageInfo;

      const startPos = getRelativeClientPos(event.nativeEvent, containerEl, imageInfo);
      const _minWidth = cropOptions.minWidth || 0;
      const _minHeight = cropOptions.minHeight || 0;

      startPos.x = clamp(startPos.x, 0, size.width - _minWidth);
      startPos.y = clamp(startPos.y, 0, size.height - _minHeight);

      const startCrop: Rect = {
        top: startPos.y,
        left: startPos.x,
        width: _minWidth,
        height: _minHeight,
      };

      initMouseMoveEvents({
        ord: 'se',
        startPos,
        startCrop,
        containerEl,
        cropOptions,
        imageInfo,
        onChange: (cropRect) => {
          setCropSelection(cropEl, imageInfo, cropRect);

          if (onCropChange) {
            onCropChange(cropRect);
          }
        },
        onComplete: (cropRect) => {
          if (startCrop !== cropRect) {
            cropRef.current = cropRect;

            setCropSelection(cropEl, imageInfo, cropRect);

            if (onCropComplete) {
              onCropComplete(cropRect);
            }
          }
        },
      });
    },
    [cropOptions, disabled, locked, keepSelection, onCropComplete, onCropChange],
  );

  const handleCropMouseDown = useCallback(
    (event: React.MouseEvent<HTMLDivElement> | React.TouchEvent<HTMLDivElement>) => {
      const containerEl = containerElRef.current;
      const cropSelectEl = cropSelectRef.current;
      const mediaEl = mediaElRef.current;

      if (disabled || !containerEl || !cropSelectEl || !mediaEl) {
        return;
      }

      event.preventDefault();
      cropSelectEl.focus({ preventScroll: true });
      imageInfoRef.current = getImageInfo(containerEl, mediaEl);
      const imageInfo = imageInfoRef.current;

      const ord = (event.target as HTMLElement).dataset.ord as CropOrd;
      const startPos = getRelativeClientPos(event.nativeEvent, containerEl, imageInfo);

      initMouseMoveEvents({
        ord,
        startPos,
        startCrop: { ...cropRef.current },
        containerEl,
        cropOptions,
        imageInfo,
        onChange: (cropRect) => {
          setCropSelection(cropSelectEl, imageInfo, cropRect);

          if (onCropChange) {
            onCropChange(cropRect);
          }
        },
        onComplete: (cropRect) => {
          cropRef.current = cropRect;
          setCropSelection(cropSelectEl, imageInfo, cropRect);

          if (onCropComplete) {
            onCropComplete(cropRect);
          }
        },
      });
    },
    [cropOptions, disabled, onCropComplete, onCropChange],
  );

  const handleCropKeyDown = useCallback(
    (e: React.KeyboardEvent<HTMLDivElement>) => {
      const cropEl = cropSelectRef.current;
      const imageInfo = imageInfoRef.current;

      if (disabled || !cropEl) {
        return;
      }

      let nudged = false;

      const nudgeStepSmall = 1;
      const nudgeStepLarge = 10;
      const nudgeStep = e.shiftKey ? nudgeStepLarge : nudgeStepSmall;
      const delta: Point = getInitialPoint();

      switch (e.key) {
        case 'ArrowLeft': {
          delta.x -= nudgeStep;
          nudged = true;
          break;
        }

        case 'ArrowRight': {
          delta.x += nudgeStep;
          nudged = true;
          break;
        }

        case 'ArrowUp': {
          delta.y -= nudgeStep;
          nudged = true;
          break;
        }

        case 'ArrowDown': {
          delta.y += nudgeStep;
          nudged = true;
          break;
        }

        default:
      }

      if (nudged) {
        e.preventDefault();

        cropRef.current = moveRect(
          {
            top: 0,
            left: 0,
            ...imageInfo.size,
          },
          cropRef.current,
          delta,
        );
        setCropSelection(cropEl, imageInfo, cropRef.current);

        if (onCropComplete) {
          onCropComplete(cropRef.current);
        }
      }
    },
    [disabled, onCropComplete],
  );

  useEffect(() => {
    function handleResize() {
      const containerEl = containerElRef.current;
      const mediaEl = mediaElRef.current;
      const cropEl = cropSelectRef.current;

      if (containerEl && mediaEl && cropEl) {
        imageInfoRef.current = getImageInfo(containerEl, mediaEl);

        setCropSelection(cropEl, imageInfoRef.current, cropRef.current);
      }
    }

    handleResize();

    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
    };
  }, [src]);

  useEffect(() => {
    if (crop && crop !== cropRef.current && cropSelectRef.current) {
      cropRef.current = crop;
      setCropSelection(cropSelectRef.current, imageInfoRef.current, cropRef.current);
    }
  }, [crop]);

  const classNames = cn(
    styles.root,
    disabled && styles.disabled,
    locked && styles.locked,
    aspectRatio && styles.fixedAspect,
    circularShape && styles.circularShape,
    keepSelection && styles.keepSelection,
    className,
  );

  return (
    <div
      ref={containerElRef}
      className={classNames}
      onMouseDown={handleContainerMouseDown}
      onTouchStart={handleContainerMouseDown}
    >
      <img
        ref={mediaElRef}
        className={styles.image}
        src={src}
        onLoad={handleImageLoad}
        alt=""
        {...other}
      />

      <div
        ref={cropSelectRef}
        className={styles.selection}
        tabIndex={-1}
        onMouseDown={handleCropMouseDown}
        onTouchStart={handleCropMouseDown}
        onKeyDown={handleCropKeyDown}
      >
        {!disabled && !locked && (
          <>
            {guides && (
              <div className={styles.guides}>
                <div className={styles.horizontalGuide} />
                <div className={styles.verticalGuide} />
                <div className={styles.cropperCenter} />
              </div>
            )}

            <div className={styles.cropElements}>
              {ORD.map((ord) => (
                <div key={`ord-${ord}`} className={styles.selectionHandle} data-ord={ord} />
              ))}
            </div>
          </>
        )}
      </div>
      {children}
    </div>
  );
};
