import { BoxSize, CropOptions, CropOrd, ImageInfo, Point, Rect } from './CropImage.types';

export function cn(...args: unknown[]) {
  return args.filter(Boolean).join(' ');
}

export function clamp(num: number, min: number, max: number) {
  return Math.min(Math.max(num, min), max);
}

export function isTouchEvent(arg: MouseEvent | TouchEvent): arg is TouchEvent {
  return 'touches' in arg && Boolean(arg.touches);
}

export function getAbsoluteClientPos(event: MouseEvent | TouchEvent): Point {
  if (isTouchEvent(event)) {
    const { pageX, pageY } = event.touches[0];

    return {
      x: pageX,
      y: pageY,
    };
  }

  return {
    x: event.pageX,
    y: event.pageY,
  };
}

export function getRelativeClientPos(
  event: MouseEvent | TouchEvent,
  element: HTMLElement,
  imageInfo: ImageInfo,
) {
  const { scale, offset } = imageInfo;

  const pos = scalePoint(getRelativePoint(getAbsoluteClientPos(event), element), scale);

  pos.x -= offset.x;
  pos.y -= offset.y;

  return pos;
}

export function getRelativePoint(point: Point, container: HTMLElement): Point {
  const { left, top } = container.getBoundingClientRect();

  return {
    x: point.x - left,
    y: point.y - top,
  };
}

function scalePoint(point: Point, scale: number): Point {
  return {
    x: Math.round(point.x / scale),
    y: Math.round(point.y / scale),
  };
}

export function getDeltaPoint(start: Point, end: Point): Point {
  return {
    x: end.x - start.x,
    y: end.y - start.y,
  };
}

export function setInCenter(container: Rect, rect: Rect): Rect {
  const top = container.top + container.height / 2 - rect.height / 2;
  const left = container.left + container.width / 2 - rect.width / 2;

  return {
    ...rect,
    top,
    left,
  };
}

export function makeAspectRect(container: Rect, rect: Rect, aspect: number): Rect {
  const completeRect: Rect = {
    ...rect,
  };

  if (rect.width) {
    completeRect.height = completeRect.width / aspect;
  }

  if (rect.height) {
    completeRect.width = completeRect.height * aspect;
  }

  if (completeRect.top + completeRect.height > container.top + container.height) {
    completeRect.height = container.top + container.height - completeRect.top;
    completeRect.width = completeRect.height * aspect;
  }

  if (completeRect.left + completeRect.width > container.left + container.width) {
    completeRect.width = container.left + container.width - completeRect.left;
    completeRect.height = completeRect.width / aspect;
  }

  completeRect.width = Math.round(completeRect.width);
  completeRect.height = Math.round(completeRect.height);

  return completeRect;
}

export function setElementPosition(el: HTMLElement, crop: Rect) {
  const { top, left, width, height } = crop;

  /* eslint-disable no-param-reassign */
  el.style.top = `${top}px`;
  el.style.left = `${left}px`;
  el.style.width = `${width}px`;
  el.style.height = `${height}px`;
  /* eslint-enable no-param-reassign */
}

export function getRectFromPoints(p1: Point, p2: Point): Rect {
  const [x1, y1, x2, y2] = [
    Math.min(p1.x, p2.x),
    Math.min(p1.y, p2.y),
    Math.max(p1.x, p2.x),
    Math.max(p1.y, p2.y),
  ];

  return {
    left: x1,
    top: y1,
    width: x2 - x1,
    height: y2 - y1,
  };
}

export function fromPoint(
  point: Point,
  width: number,
  height: number,
  quad: 'br' | 'tl' | 'tr' | 'bl' = 'br',
): Point {
  switch (quad) {
    case 'br':
      return {
        x: point.x + width,
        y: point.y + height,
      };
    case 'bl':
      return {
        x: point.x - width,
        y: point.y + height,
      };
    case 'tl':
      return {
        x: point.x - width,
        y: point.y - height,
      };
    case 'tr':
    default:
      return {
        x: point.x + width,
        y: point.y - height,
      };
  }
}

export function getMaxRect(rect: Rect, aspect: number): Rect {
  const newRect = { ...rect };

  if (rect.width / rect.height > aspect) {
    newRect.width = Math.round(rect.height * aspect);
  } else {
    newRect.height = Math.round(rect.width / aspect);
  }

  return newRect;
}

export function inverseOrd(ord: CropOrd): CropOrd {
  switch (ord) {
    case 'n': {
      return 's';
    }

    case 'ne': {
      return 'sw';
    }

    case 'e': {
      return 'w';
    }

    case 'se': {
      return 'nw';
    }

    case 's': {
      return 'n';
    }

    case 'sw': {
      return 'ne';
    }

    case 'w': {
      return 'e';
    }

    case 'nw':
    default: {
      return 'se';
    }
  }
}

export function getPointByOrd(crop: Rect, ord: CropOrd): Point {
  const right = crop.left + crop.width;
  const bottom = crop.top + crop.height;

  switch (ord) {
    case 'n': {
      return { x: crop.left, y: crop.top };
    }

    case 's': {
      return { x: right, y: bottom };
    }

    case 'e': {
      return { x: right, y: bottom };
    }

    case 'w': {
      return { x: crop.left, y: crop.top };
    }

    case 'se': {
      return { x: right, y: bottom };
    }

    case 'sw': {
      return { x: crop.left, y: bottom };
    }

    case 'ne': {
      return { x: right, y: crop.top };
    }

    case 'nw':
    default: {
      return { x: crop.left, y: crop.top };
    }
  }
}

export function getDragQuadrant(lockedPoint: Point, nextPoint: Point) {
  const relX = lockedPoint.x - nextPoint.x;
  const relY = lockedPoint.y - nextPoint.y;

  if (relX < 0 && relY < 0) {
    return 'br';
  }

  if (relX >= 0 && relY >= 0) {
    return 'tl';
  }

  if (relX < 0 && relY >= 0) {
    return 'tr';
  }

  return 'bl';
}

export function moveRect(container: Rect, rect: Rect, delta: Point): Rect {
  const top = clamp(
    rect.top + delta.y,
    container.top,
    container.top + container.height - rect.height,
  );
  const left = clamp(
    rect.left + delta.x,
    container.left,
    container.left + container.width - rect.width,
  );

  return {
    ...rect,
    top,
    left,
  };
}

export function resizeRect(
  container: Rect,
  rect: Rect,
  delta: Point,
  ord: CropOrd,
  options: CropOptions,
): Rect {
  const { aspect } = options;
  const locked = getPointByOrd(rect, inverseOrd(ord));
  const stuck = getPointByOrd(rect, ord);

  const containerMaxWidth = container.left + container.width;
  const containerMaxHeight = container.top + container.height;
  const maxWidth = options.maxWidth || containerMaxWidth;
  const maxHeight = options.maxHeight || containerMaxHeight;
  let minWidth = options.minWidth || 0;
  let minHeight = options.minHeight || 0;

  const newPoint: Point = {
    x: clamp(stuck.x + delta.x, container.left, containerMaxWidth),
    y: clamp(stuck.y + delta.y, container.top, containerMaxHeight),
  };

  if (ord === 'n' || ord === 's') {
    newPoint.x = stuck.x;
  }

  if (ord === 'e' || ord === 'w') {
    newPoint.y = stuck.y;
  }

  if (aspect) {
    minWidth = Math.round(Math.max(minWidth, minHeight * aspect));
    minHeight = Math.round(Math.max(minHeight, minWidth / aspect));
  }

  if (locked.x > newPoint.x) {
    newPoint.x = clamp(newPoint.x, locked.x - maxWidth, locked.x - minWidth);
  } else {
    newPoint.x = clamp(newPoint.x, locked.x + minWidth, locked.x + maxWidth);
  }

  if (locked.y < newPoint.y) {
    newPoint.y = clamp(newPoint.y, locked.y + minHeight, locked.y + maxHeight);
  } else {
    newPoint.y = clamp(newPoint.y, locked.y - maxHeight, locked.y - minHeight);
  }

  if (locked.x - minWidth < 0) {
    newPoint.x = Math.max(newPoint.x, locked.x + minWidth);
  }

  if (locked.x + minWidth > containerMaxWidth) {
    newPoint.x = Math.min(newPoint.x, locked.x - minWidth);
  }

  if (locked.y - minHeight < 0) {
    newPoint.y = Math.max(newPoint.y, locked.y + minHeight);
  }

  if (locked.y + minHeight > containerMaxHeight) {
    newPoint.y = Math.min(newPoint.y, locked.y - minHeight);
  }

  if (aspect) {
    const quad = getDragQuadrant(locked, newPoint);
    const maxRect = getMaxRect(getRectFromPoints(locked, newPoint), aspect);
    const { x, y } = fromPoint(locked, maxRect.width, maxRect.height, quad);

    newPoint.x = x;
    newPoint.y = y;
  }

  const crop = getRectFromPoints(locked, newPoint);

  if (ord === 'e' || ord === 'w') {
    const mY = rect.top + rect.height / 2;

    crop.top = mY - crop.height / 2;
  }

  if (ord === 'n' || ord === 's') {
    const mX = rect.left + rect.width / 2;

    crop.left = mX - crop.width / 2;
  }

  return crop;
}

export function getImageInfo(container: HTMLElement, image: HTMLImageElement) {
  const containerRect = container.getBoundingClientRect();
  const width = Math.round(containerRect.width);
  const height = Math.round(containerRect.height);
  const { naturalWidth, naturalHeight } = image;

  const size: BoxSize = {
    width: naturalWidth,
    height: naturalHeight,
  };

  const scale = Math.min(width / size.width, height / size.height);
  const newWidth = Math.round(width / scale);
  const newHeight = Math.round(height / scale);

  const offset: Point = {
    x: (newWidth - size.width) / 2,
    y: (newHeight - size.height) / 2,
  };

  return {
    scale,
    size,
    offset,
  };
}

export function scaleRect(rect: Rect, scale: number): Rect {
  return {
    top: rect.top * scale,
    left: rect.left * scale,
    width: Math.round(rect.width * scale),
    height: Math.round(rect.height * scale),
  };
}

export function setCropSelection(element: HTMLElement, imageInfo: ImageInfo, crop: Rect) {
  const { scale, offset } = imageInfo;
  const rect = { ...crop };

  rect.left += offset.x;
  rect.top += offset.y;

  requestAnimationFrame(() => {
    setElementPosition(element, scaleRect(rect, scale));
  });
}

export function makeDefaultCrop(imageInfo: ImageInfo, options: CropOptions) {
  const { size } = imageInfo;

  const container: Rect = {
    top: 0,
    left: 0,
    width: size.width,
    height: size.height,
  };

  let nextCrop: Rect = {
    top: 0,
    left: 0,
    ...size,
  };

  if (options.aspect) {
    nextCrop = makeAspectRect(container, nextCrop, options.aspect);
  }

  return setInCenter(container, nextCrop);
}

export function isEqualRect(first: Rect, second: Rect) {
  return (
    first.top === second.top &&
    first.left === second.left &&
    first.width === second.width &&
    first.height === second.height
  );
}

export function getInitialRect(): Rect {
  return {
    top: 0,
    left: 0,
    width: 0,
    height: 0,
  };
}

export function getInitialPoint(): Point {
  return {
    x: 0,
    y: 0,
  };
}

export interface InitMouseMoveEventsOptions {
  containerEl: HTMLElement;
  startPos: Point;
  imageInfo: ImageInfo;
  startCrop: Rect;
  ord?: CropOrd;
  cropOptions: CropOptions;
  onChange: (crop: Rect) => void;
  onComplete: (crop: Rect) => void;
}

export function initMouseMoveEvents(options: InitMouseMoveEventsOptions) {
  const { containerEl, cropOptions, ord, startPos, imageInfo, startCrop, onChange, onComplete } =
    options;
  const { size } = imageInfo;
  const imageRect = {
    top: 0,
    left: 0,
    ...size,
  };

  let lastCrop = startCrop;

  function handleMove(e: MouseEvent | TouchEvent) {
    const endPos = getRelativeClientPos(e, containerEl, imageInfo);
    const delta = getDeltaPoint(startPos, endPos);

    let nextCrop: Rect;

    if (typeof ord === 'string') {
      const clonedCropOptions = { ...cropOptions };

      if (e.shiftKey && !clonedCropOptions.aspect) {
        clonedCropOptions.aspect = lastCrop.width / lastCrop.height;
      }

      nextCrop = resizeRect(imageRect, startCrop, delta, ord, clonedCropOptions);
    } else {
      nextCrop = moveRect(imageRect, startCrop, delta);
    }

    if (!isEqualRect(lastCrop, nextCrop)) {
      lastCrop = nextCrop;
      onChange(lastCrop);
    }
  }

  function handleUp() {
    document.removeEventListener('pointermove', handleMove);
    document.removeEventListener('pointerup', handleUp);
    document.removeEventListener('touchend', handleUp);
    document.removeEventListener('touchmove', handleMove);
    document.removeEventListener('touchcancel', handleUp);

    if (!isEqualRect(lastCrop, startCrop)) {
      onComplete(lastCrop);
    }
  }

  document.addEventListener('pointermove', handleMove);
  document.addEventListener('mouseup', handleUp);
  document.addEventListener('touchend', handleUp);
  document.addEventListener('touchmove', handleMove);
  document.addEventListener('touchcancel', handleUp);
}
