import cx from 'classnames';
import { ElementType, ReactElement, ReactNode, VFC } from 'react';

import { ListItemView, ListItemViewProps, Text } from '@yandex-id/components';

import styles from './List.module.css';

export type PlateColor =
  | 'default'
  | 'warning'
  | 'blue-gradient'
  | 'violet-gradient100'
  | 'green-gradient100'
  | 'red-gradient100'
  | 'blue-gradient100';

export interface ListItemProps<T extends ElementType = 'div'> extends ListItemViewProps<T> {
  after?: ReactNode;
  counter?: number;
  description?: string;
  icon: ReactNode;
  label?: string;
  text: string;
  warning?: string | boolean;
  plateColor?: PlateColor;
}

export interface ListItemComponent extends Pick<VFC, keyof VFC> {
  <T extends ElementType = 'div'>(props: ListItemProps<T>): ReactElement | null;
}

export const ListItem: ListItemComponent = (props) => {
  const {
    icon,
    text,
    label,
    counter,
    warning,
    description,
    after,
    plateColor = 'default',
    tabIndex = 0,
    role = 'button',
    ...restProps
  } = props;

  return (
    <ListItemView tabIndex={tabIndex} role={role} {...restProps}>
      <ListItemView.Before>
        <div className={cx(styles.plate, styles[plateColor])}>{icon}</div>
      </ListItemView.Before>
      <ListItemView.Content alignItems="center">
        {label && (
          <Text variant="text-s" color="secondary" className={styles.label}>
            {label}
            {warning && <span className={styles.hint}> • {warning}</span>}
          </Text>
        )}
        <Text variant="text-m">{text}</Text>
        {description && (
          <Text variant="text-s" color="secondary" className={styles.description}>
            {description}
          </Text>
        )}
      </ListItemView.Content>
      <ListItemView.After className={styles.slotAfter}>
        {Boolean(counter) && (
          <span className={styles.counter}>
            <Text variant="text-s" color="inherit">
              {counter}
            </Text>
          </span>
        )}
        {after}
      </ListItemView.After>
    </ListItemView>
  );
};
