import cx from 'classnames';
import { FC, HTMLAttributes } from 'react';

import { Text } from '@yandex-id/components';
import { createSlot, useSlots } from '@yandex-id/components/libs/slots';

import { Title, TitleProps } from './Title';

import styles from './Section.module.css';

interface SectionProps {
  className?: string;
}

interface TitleSlotProps extends TitleProps {
  align?: 'start' | 'center' | 'end';
}

interface DescriptionProps {
  isSmall?: boolean;
}

const MediaSlot = createSlot('Section.Media');
const TitleSlot = createSlot<TitleSlotProps>('Section.Title');
const DescriptionSlot = createSlot<DescriptionProps>('Section.Description');
const AsideSlot = createSlot('Section.Aside');
const ContentSlot = createSlot<HTMLAttributes<HTMLElement>>('Section.Content');

const _Section: FC<SectionProps> = (props) => {
  const slots = useSlots(props, { defaultSlot: ContentSlot });

  const media = slots.get(MediaSlot);
  const title = slots.get(TitleSlot);
  const description = slots.get(DescriptionSlot);
  const aside = slots.get(AsideSlot);
  const content = slots.get(ContentSlot);

  return (
    <div className={cx(styles.root, props.className)}>
      {media && <div className={styles.media}>{media.rendered}</div>}
      {title && (
        <div className={cx(styles.heading, styles[`heading-align_${title.props.align}`])}>
          <Title
            anchor={title.props.anchor}
            isHeading={title.props.isHeading}
            variant={title.props.variant}
            weight={title.props.weight}
          >
            {title.rendered}
          </Title>
          {description && (
            <Text
              variant={description.props.isSmall ? 'text-s' : 'text-m'}
              className={styles.description}
            >
              {description.rendered}
            </Text>
          )}
          {aside && <div className={styles.aside}>{aside.rendered}</div>}
        </div>
      )}
      <div {...content?.props} className={cx(styles.content, content?.props.className)}>
        {slots.children}
      </div>
    </div>
  );
};

export const Section = Object.assign(_Section, {
  Media: MediaSlot,
  Title: TitleSlot,
  Description: DescriptionSlot,
  Aside: AsideSlot,
  Content: ContentSlot,
});
