import cx from 'classnames';
import { ChangeEvent, ReactNode, RefObject, VFC, useCallback, useMemo, useRef } from 'react';

import { mergeProps, useFocusable, useForkRef, useLabel } from '@use-platform/react';
import { TextInput, TextInputProps } from '@yandex-id/components';

import styles from './TextField.module.css';
import sizeXl from './_size/size-xl.module.css';

export interface TextFieldProps extends Omit<TextInputProps, 'addonBefore' | 'addonAfter'> {
  after?: ReactNode;
  before?: ReactNode;
  error?: string | object | null;
  inputRef?: RefObject<HTMLInputElement>;
  label?: string;
  onChangeValue?: (value: string) => void;
}

export const TextField: VFC<TextFieldProps> = (props) => {
  const {
    after,
    before,
    className,
    disabled: isDisabled,
    error,
    inputRef: controlRef,
    label,
    onChange,
    onChangeValue,
    size,
    ...restProps
  } = props;
  const { labelProps, fieldProps } = useLabel();
  const inputRef = useRef<HTMLInputElement>(null);
  const forkedInputRef = useForkRef(inputRef, controlRef);
  const isInvalid = Boolean(error);

  useFocusable(props, inputRef);

  const handleOnChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      onChange?.(event);
      onChangeValue?.(event.target.value);
    },
    [onChange, onChangeValue],
  );

  const addonBefore = useMemo(() => {
    return before ? <span className={styles.addon}>{before}</span> : undefined;
  }, [before]);

  const addonAfter = useMemo(() => {
    return after ? <span className={styles.addon}>{after}</span> : undefined;
  }, [after]);

  return (
    <div
      className={cx(
        styles.root,
        {
          // NOTE: Временное решение, в будущем нужно использовать css-modules compose API.
          [sizeXl.root]: size === 'xl',
          [styles.root_isDisabled]: isDisabled,
          [styles.root_isInvalid]: isInvalid,
        },
        className,
      )}
    >
      {label && (
        <label {...labelProps} className={styles.label}>
          {label}
        </label>
      )}
      <TextInput
        {...mergeProps(restProps, fieldProps)}
        className={styles.control}
        controlRef={forkedInputRef}
        disabled={isDisabled}
        iconLeft={addonBefore}
        iconRight={addonAfter}
        onChange={handleOnChange}
        size={size}
      />
      {error && <span className={styles.error}>{error}</span>}
    </div>
  );
};
