import cx from 'classnames';
import React, { forwardRef } from 'react';

import { CameraOutline, ChildProfile, EditOutline } from '@client/shared/icons';
import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';
import { Text } from '@yandex-id/components';

import { i18n } from './i18n';

import styles from './UserInfo.module.css';

export interface UserInfoProps extends PressProps {
  avatar?: string;
  displayName: string;
  fullName?: string;
  birthdate?: string;
  city?: string;
  hasPlus?: boolean;
  isChild?: boolean;
  tabIndex?: number;
}

export const UserInfo = forwardRef<HTMLDivElement, UserInfoProps>((props, ref) => {
  const {
    avatar,
    displayName,
    fullName,
    birthdate,
    city,
    hasPlus,
    isChild,
    tabIndex = 0,
    onPress,
    onPressStart,
    onPressUp,
    onPressEnd,
    ...restProps
  } = props;
  const { pressProps } = usePress(props);
  const { isHovered, hoverProps } = useHover({});

  return (
    <div
      ref={ref}
      tabIndex={tabIndex}
      className={cx(styles.root, {
        [styles.root_isHovered]: isHovered,
        [styles.root_hasFullName]: Boolean(fullName),
      })}
      role="button"
      {...mergeProps(hoverProps, pressProps)}
      {...restProps}
    >
      <div
        className={cx(styles.avatarWrap, {
          [styles.avatarWrap_hasPlus]: hasPlus,
        })}
      >
        <div className={styles.avatarMask}>
          {!avatar && <CameraOutline />}
          {avatar && <img src={avatar} className={styles.avatar} />}
        </div>
        {isChild && (
          <div className={styles.kidsBadge}>
            <ChildProfile />
          </div>
        )}
      </div>
      <div className={styles.displayName}>
        <Text variant="text-m">{fullName ?? displayName}</Text>
      </div>
      <div className={styles.fullName}>
        {!fullName && (
          <Text variant="text-s" color="negative">
            {i18n('Нажмите, чтобы добавить ФИО', {
              context:
                'Плейсхолдер отсутствующего ФИО в блоке информации о пользователе на странице данных в Паспорте',
            })}
          </Text>
        )}
        {fullName && (
          <Text variant="text-s" color="secondary" as="div">
            {i18n('Зовите меня:', {
              context: 'Лейбл публичного имени в виджете профиля в Паспорте',
            })}{' '}
            {displayName}
            {(birthdate || city) && <br />}
            {birthdate}
            {birthdate && city && ' • '}
            {city}
          </Text>
        )}
      </div>
      <EditOutline className={styles.moreIcon} />
    </div>
  );
});
