import { observer } from 'mobx-react-lite';
import { VFC, useCallback, useMemo, useState } from 'react';

import { useStore } from '@client/shared/libs/mobx-next';
import { Section } from '@client/shared/ui-kit';
import { AccountAction } from '@shared/types/account';

import { AccountActionsStore } from './AccountActions.model';
import { ActionDialog } from './ActionDialog';
import { ActionGroup } from './ActionGroup';
import { ActionPressEvent } from './ActionItem';
import { EndOfTime } from './EndOfTime';
import { AccountActionsSkeleton } from './Skeleton';
import { i18n } from './i18n';
import { groupByDate } from './utils';

const AccountActionsList = observer(() => {
  const { data, isLoading } = useStore(AccountActionsStore);

  const grouped = useMemo(() => {
    return data ? groupByDate(data) : [];
  }, [data]);

  const [selectedAction, setSelectedAction] = useState<AccountAction | null>(null);

  const handleAction = useCallback((event: ActionPressEvent) => {
    setSelectedAction(event.data);
  }, []);

  const handleClose = useCallback(() => {
    setSelectedAction(null);
  }, []);

  if (isLoading) {
    return <AccountActionsSkeleton />;
  }

  return (
    <>
      {grouped.map((group, index) => (
        <ActionGroup
          key={index}
          title={group.title}
          actions={group.actions}
          onPressAction={handleAction}
        />
      ))}

      <EndOfTime />
      {selectedAction && <ActionDialog action={selectedAction} onClose={handleClose} />}
    </>
  );
});

export const AccountActions: VFC = () => {
  return (
    <Section>
      <Section.Title isHeading>{i18n('Активность')}</Section.Title>
      <Section.Description>
        {i18n(
          'Если вы видите что-то подозрительное — выберите событие и нажмите кнопку «Это не я».',
        )}
      </Section.Description>

      <AccountActionsList />
    </Section>
  );
};
