import { useRouter } from 'next/router';
import { VFC, useCallback, useState } from 'react';

import { usePassportPaths } from '@client/shared/paths';
import { Dialog } from '@client/shared/ui-kit';
import { AccountAction } from '@shared/types/account';

import { ActionInfo, ActionInfoEvent, ActionInfoEventType } from '../ActionInfo';
import { ItWasNotMe, ItWasNotMeEvent, ItWasNotMeEventType } from '../ItWasNotMe';

import styles from './ActionDialog.module.css';

enum Screen {
  ACTION_INFO = 'ACTION_INFO',
  IT_WAS_NOT_ME = 'IT_WAS_NOT_ME',
}

export interface ActionDialogProps {
  action: AccountAction;
  onClose: () => void;
}

export const ActionDialog: VFC<ActionDialogProps> = (props) => {
  const { action, onClose } = props;
  const router = useRouter();
  const [screen, setScreen] = useState<Screen>(Screen.ACTION_INFO);
  const { changePassword } = usePassportPaths();

  const handleEvent = useCallback(
    (event: ActionInfoEvent | ItWasNotMeEvent) => {
      switch (event.type) {
        case ActionInfoEventType.CLOSE:
          onClose();
          break;

        case ActionInfoEventType.IT_WAS_NOT_ME:
          setScreen(Screen.IT_WAS_NOT_ME);
          break;

        case ItWasNotMeEventType.FINE:
          setScreen(Screen.ACTION_INFO);
          break;

        case ItWasNotMeEventType.CHANGE_PASSWORD:
          router.push(changePassword);
          break;
      }
    },
    [changePassword, onClose, router],
  );

  const handleClose = useCallback(() => {
    switch (screen) {
      case Screen.ACTION_INFO:
        handleEvent({ type: ActionInfoEventType.CLOSE });
        break;

      case Screen.IT_WAS_NOT_ME:
        handleEvent({ type: ItWasNotMeEventType.FINE });
        break;
    }
  }, [handleEvent, screen]);

  return (
    <Dialog hasClose={false} visible onClose={handleClose}>
      <div className={styles.content}>
        {screen === Screen.ACTION_INFO && (
          <ActionInfo action={action} onActionEvent={handleEvent} />
        )}

        {screen === Screen.IT_WAS_NOT_ME && <ItWasNotMe onActionEvent={handleEvent} />}
      </div>
    </Dialog>
  );
};
