import cn from 'classnames';
import { VFC } from 'react';

import {
  IdOutline,
  KeyCross,
  KeyPlus,
  KeyRefresh,
  Kvko,
  Login,
  MailCross,
  MailPlus,
  PhoneCross,
  PhonePlus,
  PhoneRefresh,
  Profile,
  QuestionAnswerCross,
  Refresh,
} from '@client/shared/icons';
import { Browser, BrowserName } from '@client/shared/icons/browsers';
import { AccountAction, AccountActionType, AccountAuthType } from '@shared/types/account';

import styles from './ActionIcon.module.css';

type IconSize = 12 | 16 | 24 | 32;

interface IconProps {
  className?: string;
  size?: IconSize;
}

const ICON_BY_TYPE: Record<AccountActionType, VFC<IconProps>> = {
  [AccountActionType.PasswordChange]: KeyRefresh,
  [AccountActionType.PersonalData]: Profile,
  [AccountActionType.SecurePhoneSet]: PhonePlus,
  [AccountActionType.SecurePhoneUnset]: PhoneCross,
  [AccountActionType.SecurePhoneReplace]: PhoneRefresh,
  [AccountActionType.PhoneBind]: PhonePlus,
  [AccountActionType.PhoneUnbind]: PhoneCross,
  [AccountActionType.AppPasswordEnabled]: KeyPlus,
  [AccountActionType.AppPasswordsDisabled]: KeyCross,
  [AccountActionType.EmailAdd]: MailPlus,
  [AccountActionType.EmailRemove]: MailCross,
  [AccountActionType.EmailRemoveAll]: MailCross,
  [AccountActionType.TotpEnabled]: IdOutline,
  [AccountActionType.TotpDisabled]: IdOutline,
  [AccountActionType.TotpMigrated]: IdOutline,
  [AccountActionType.Restore]: Refresh,
  [AccountActionType.GlobalLogout]: IdOutline,
  [AccountActionType.QuestionsChange]: Kvko,
  [AccountActionType.QuestionsRemove]: QuestionAnswerCross,
  [AccountActionType.Login]: Login,
  [AccountActionType.PasswordRemove]: KeyCross,
  [AccountActionType.WebSessionsRevoked]: IdOutline,
  [AccountActionType.TokensRevoked]: IdOutline,
  [AccountActionType.AppPasswordsRevoked]: IdOutline,
};

const BROWSERS: Record<string, BrowserName | undefined> = {
  Firefox: 'firefox',
  MobileFirefox: 'firefox-mobile',
  MobileSafari: 'safari-mobile',
  Opera: 'opera',
  OperaMobile: 'opera-mobile',
  AndroidBrowser: 'android',
  YandexBrowser: 'yandex',
  Safari: 'safari',
  MSIE: 'msie',
  Edge: 'edge',
  Chrome: 'chrome',
  ChromeMobile: 'chrome-mobile',
  Chromium: 'chromium',
  'Samsung Internet': 'samsung-internet',
  YandexSearch: 'yandex',
  YandexBrowserLite: 'yandex-lite',
};

function isNegativeAction(actionType: AccountActionType) {
  return [
    AccountActionType.EmailRemove,
    AccountActionType.EmailRemoveAll,
    AccountActionType.PhoneUnbind,
    AccountActionType.SecurePhoneUnset,
    AccountActionType.PasswordRemove,
    AccountActionType.QuestionsRemove,
    AccountActionType.AppPasswordsDisabled,
  ].includes(actionType);
}

export interface ActionIconProps extends IconProps {
  action: AccountAction;
}

export const ActionIcon: VFC<ActionIconProps> = (props) => {
  const { action, size, className } = props;
  const { browser } = action;

  if (action.type === AccountActionType.Login && action.authType === AccountAuthType.Web) {
    const browserName = browser ? BROWSERS[browser.name] : undefined;

    if (browserName) {
      return <Browser className={className} name={browserName} size={size} />;
    }
  }

  const Icon = ICON_BY_TYPE[action.type] || IdOutline;

  return (
    <Icon
      className={cn(className, { [styles.negative]: isNegativeAction(action.type) })}
      size={size}
    />
  );
};
