import { VFC, useCallback } from 'react';

import { ErrorOutline } from '@client/shared/icons';
import { useTimestampFormat } from '@client/shared/libs/use-timestamp-format';
import { Button, CardTemplate, Text } from '@client/shared/ui-kit';
import { AccountAction, AccountActionType, AccountRestoreByType } from '@shared/types/account';

import { ActionIcon } from '../ActionIcon';
import { getActionText } from '../ActionItem/i18nHelpers';
import { PersonalFields } from '../PersonalFields';
import dummyMap from './assets/dummy-map.svg';
import { i18n } from './i18n';

import styles from './ActionInfo.module.css';

const RESTORE_BY_TEXTS: Record<AccountRestoreByType, () => string> = {
  [AccountRestoreByType.SemiAuto]: () => i18n('через анкету восстановления'),
  [AccountRestoreByType.Link]: () => i18n('через службу поддержки'),
  [AccountRestoreByType.PhoneAnd2faFactor]: () => i18n('через 2fa'),
  [AccountRestoreByType.Email]: () => i18n('через почту'),
  [AccountRestoreByType.Phone]: () => i18n('через телефон'),
  [AccountRestoreByType.Hint]: () => i18n('через контрольный вопрос'),
};

export enum ActionInfoEventType {
  CLOSE = 'close',
  IT_WAS_NOT_ME = 'it-was-not-me',
}

export interface ActionInfoEvent {
  type: ActionInfoEventType;
}

export interface ActionInfoProps {
  action: AccountAction;
  onActionEvent: (event: ActionInfoEvent) => void;
}

export const ActionInfo: VFC<ActionInfoProps> = (props) => {
  const { action, onActionEvent } = props;
  const { timestamp, geolocation, os, browser } = action;
  const { region, ip, mapUrl = dummyMap.src as string } = geolocation;

  const formattedDate = useTimestampFormat(timestamp);
  const browserName = browser ? `${browser.name} ${browser.version}` : null;
  const osName = os ? `${os.name} ${os.version}` : null;

  const handleClose = useCallback(() => {
    onActionEvent({ type: ActionInfoEventType.CLOSE });
  }, [onActionEvent]);

  const handleItWasNotMe = useCallback(() => {
    onActionEvent({ type: ActionInfoEventType.IT_WAS_NOT_ME });
  }, [onActionEvent]);

  return (
    <CardTemplate>
      <CardTemplate.Media
        className={styles.location}
        style={{ backgroundImage: `url("${mapUrl}")` }}
      >
        <div className={styles.icon}>
          <ActionIcon action={action} size={32} />
        </div>
      </CardTemplate.Media>

      <CardTemplate.Content>
        <Text as="div" variant="heading-l" weight="bold">
          {getActionText(action)}
        </Text>

        <Text as="div" variant="text-m-long">
          {action.type === AccountActionType.PersonalData && (
            <>
              <PersonalFields fields={action.changedFields} />
              <br />
            </>
          )}

          {action.type === AccountActionType.Restore && (
            <>
              {i18n('Доступ к аккаунту восстановлен')} {RESTORE_BY_TEXTS[action.restoreBy.type]()}
              <br />
              <br />
            </>
          )}

          {formattedDate}
          {region && <>, {region}</>}

          {ip && (
            <>
              <br />
              IP {ip}
            </>
          )}

          {(browserName || osName) && (
            <>
              <br />
              {browserName}
              {browserName && osName && <> – </>}
              {osName}
            </>
          )}
        </Text>
      </CardTemplate.Content>

      <CardTemplate.Actions>
        <Button size="xl" onPress={handleClose}>
          {i18n('Закрыть')}
        </Button>
        <Button variant="clear" size="xl" before={<ErrorOutline />} onPress={handleItWasNotMe}>
          {i18n('Это не я')}
        </Button>
      </CardTemplate.Actions>
    </CardTemplate>
  );
};
