import { VFC, useCallback, useState } from 'react';

import { AuthType } from '@client/shared/api/graphql';
import { Key } from '@client/shared/icons';
import { useCurrentUrl } from '@client/shared/paths';
import { ListItemRadio } from '@client/shared/ui-kit';

import {
  LowerProtectionAlert,
  LowerProtectionAlertEvent,
  LowerProtectionAlertEventType,
} from '../../LowerProtectionAlert';
import { useUpgradeAccountUrl, useYandexKeyUrls } from '../../libs';
import { PasswordAndSmsModal } from '../../model';

export interface PasswordOnlyItemProps {
  text: string;
  description: string;
  authType: AuthType | null;
}

export const PasswordOnlyItem: VFC<PasswordOnlyItemProps> = (props) => {
  const { text, description, authType } = props;
  const [visible, setVisible] = useState(false);
  const currentUrl = useCurrentUrl();
  const upgradeAccountUrl = useUpgradeAccountUrl();
  const yandexKeyUrls = useYandexKeyUrls({ retpath: currentUrl.href });

  const handlePress = useCallback(() => {
    if (authType === AuthType.PasswordOnly) {
      return;
    }

    if (authType === null) {
      window.location.href = upgradeAccountUrl;
    } else {
      setVisible(true);
    }
  }, [authType, upgradeAccountUrl]);

  const handleAction = useCallback(
    (event: LowerProtectionAlertEvent) => {
      setVisible(false);

      if (event.type === LowerProtectionAlertEventType.CONTINUE) {
        switch (authType) {
          case AuthType.SmsOnly:
            window.location.href = upgradeAccountUrl;
            break;

          case AuthType.YandexKey:
            window.location.href = yandexKeyUrls.disable;
            break;

          case AuthType.PasswordAndSms:
            PasswordAndSmsModal.show();
            break;
        }
      }
    },
    [authType, upgradeAccountUrl, yandexKeyUrls.disable],
  );

  return (
    <>
      <ListItemRadio
        icon={<Key />}
        text={text}
        description={description}
        isChecked={authType === AuthType.PasswordOnly}
        onPress={handlePress}
      />
      <LowerProtectionAlert visible={visible} onAction={handleAction} />
    </>
  );
};
