import { FC, VFC, useCallback, useEffect, useState } from 'react';

import { BankCard as BankCardData, useBankCardsQuery } from '@client/shared/api/graphql';
import { PayServiceIcon } from '@client/shared/icons';
import { useEnvContext } from '@client/shared/libs/env';
import { sendParams } from '@client/shared/libs/metrika';
import { BankCard, Carousel, Section } from '@client/shared/ui-kit';

import { i18nRaw } from './i18n';
import { unbindBankCard } from './api';
import { AddCard } from './components/AddCard';
import { Alert } from './components/Alert';
import { CardBindingForm } from './components/CardBindingForm';
import { CardSettings } from './components/CardSettings';
import { BankCardsWidgetSkeleton } from './components/Skeleton';
import { dataToBankCardProps } from './helpers';

import styles from './BankCards.module.css';

const WidgetWrap: FC = ({ children }) => {
  return (
    <Section>
      <Section.Title anchor="bank-cards">
        {i18nRaw('Карты в {icon}', {
          icon: (
            <PayServiceIcon
              key="PayServiceIcon"
              className={styles.payIcon}
              height={24}
              width={56}
            />
          ),
        })}
      </Section.Title>
      {children}
    </Section>
  );
};

const BankCardsWidget: VFC = () => {
  const [openedPopup, setOpenedPopup] = useState<string | null>(null);
  const [lastOpenedPopup, setLastOpenedPopup] = useState<string | null>(null);

  const [selectedCard, setSelectedCard] = useState<BankCardData | null>(null);

  const { data, loading, refetch } = useBankCardsQuery();

  useEffect(() => {
    sendParams('show.finance.bank-cards');
  }, []);

  const onCloseBinding = useCallback(() => {
    setOpenedPopup(null);
  }, []);

  const onCardBinded = useCallback(() => {
    sendParams('finance.bank-cards.binding.success');
    setOpenedPopup(null);
    refetch();
  }, [refetch]);

  const onCloseSettings = useCallback(() => {
    setOpenedPopup(null);
  }, []);

  const onOpenBinding = useCallback(() => {
    sendParams('click.finance.bank-cards.binding');
    setOpenedPopup('binding');
  }, []);

  const onClickCard = useCallback(
    (card: BankCardData) => () => {
      sendParams('click.finance.bank-cards.card');
      setSelectedCard(card);
      setOpenedPopup('settings');
    },
    [],
  );

  const onDeleteClick = useCallback(() => {
    setLastOpenedPopup(openedPopup);
    setOpenedPopup('alert');
  }, [openedPopup]);

  const onCloseAlert = useCallback(() => {
    setOpenedPopup(lastOpenedPopup);
  }, [lastOpenedPopup]);

  const onAcceptDelete = useCallback(() => {
    selectedCard &&
      unbindBankCard(selectedCard.id).then(() => {
        sendParams('finance.bank-cards.card.delete');
        refetch();
      });
    setOpenedPopup(null);
    setLastOpenedPopup(null);
  }, [selectedCard, refetch]);

  if (loading || !data) {
    return (
      <WidgetWrap>
        <BankCardsWidgetSkeleton />
      </WidgetWrap>
    );
  }

  if (data.accountBankCards.status === 'error') {
    return null;
  }

  return (
    <WidgetWrap>
      <Carousel>
        <AddCard onPress={onOpenBinding} />
        {data.accountBankCards.cards.map((card) => (
          <BankCard
            key={card.id}
            onPress={onClickCard(card)}
            className={styles.card}
            {...dataToBankCardProps(card)}
            size="s"
          />
        ))}
      </Carousel>

      <CardBindingForm
        trackId={data.accountBankCards.trackId}
        visible={openedPopup === 'binding'}
        onClose={onCloseBinding}
        onSuccess={onCardBinded}
      />
      <CardSettings
        cardData={selectedCard}
        visible={openedPopup === 'settings'}
        onClose={onCloseSettings}
        onDeleteClick={onDeleteClick}
      />
      <Alert visible={openedPopup === 'alert'} onClose={onCloseAlert} onAccept={onAcceptDelete} />
    </WidgetWrap>
  );
};

export const BankCards = () => {
  const { tld } = useEnvContext();

  if (tld !== 'ru') {
    return null;
  }

  return <BankCardsWidget />;
};
