import Image from 'next/image';
import { VFC } from 'react';

import CardDeleteImage from '@client/shared/assets/card-delete.png';
import PlusDeleteImage from '@client/shared/assets/plus-delete.png';
import { TrashOutline } from '@client/shared/icons';
import { Button, Dialog, PromoContent } from '@client/shared/ui-kit';

import { i18n } from './i18n';

import styles from './Alert.module.css';

type AlertType = 'plus' | 'subscription' | 'default';
type PromoTexts = {
  title: () => string;
  description: () => string;
  image: StaticImageData;
};

interface IAlertProps {
  visible: boolean;
  type?: AlertType;
  onClose(): void;
  onAccept(): void;
}

const DATA: Record<AlertType, PromoTexts> = {
  default: {
    title: () => i18n('Точно удалить карту?'),
    description: () =>
      i18n('В Такси, Еде, на Маркете и других сервисах Яндекса её данные придётся вводить заново.'),
    image: CardDeleteImage,
  },
  plus: {
    title: () => i18n('Точно удалить? Можно потерять Плюс'),
    description: () =>
      i18n('Без карты подписка не продлится — и у вас не будет фильмов, музыки и кешбэка'),
    image: PlusDeleteImage,
  },
  subscription: {
    title: () => i18n('Точно удалить? Можно потерять подписки'),
    description: () =>
      i18n('Без карты подписка не продлится — и у вас не будет фильмов, музыки и кешбэка'),
    image: CardDeleteImage,
  },
};

export const Alert: VFC<IAlertProps> = ({ type = 'default', visible, onClose, onAccept }) => {
  return (
    <Dialog visible={visible} onClose={onClose} className={styles.root}>
      <PromoContent>
        <PromoContent.Image>
          <Image layout="fixed" src={DATA[type].image} height={120} objectFit="contain" />
        </PromoContent.Image>
        <PromoContent.Title>{DATA[type].title()}</PromoContent.Title>
        <PromoContent.Description>{DATA[type].description()}</PromoContent.Description>
        <PromoContent.Button>
          <Button className={styles.button} variant="action" size="xl" onPress={onClose}>
            {i18n('Оставить карту')}
          </Button>
        </PromoContent.Button>
        <PromoContent.SecondaryButton>
          <Button variant="clear" size="xl" onPress={onAccept} before={<TrashOutline />}>
            {i18n('Удалить')}
          </Button>
        </PromoContent.SecondaryButton>
      </PromoContent>
    </Dialog>
  );
};
