import { RefObject, useCallback, useEffect, useMemo, useState } from 'react';

import { useEnvContext } from '@client/shared/libs/env';
import TrustSDK, { PaymentFrame } from '@yandex-trust-sdk/base';

interface useTrustSdkProps {
  onClose(): void;
  onSuccess(): void;
  width?: string;
  height?: string;
}

export const useTrustSdk = ({
  onClose,
  onSuccess,
  width: frameWidth,
  height: frameHeight,
}: useTrustSdkProps) => {
  const { yenv } = useEnvContext();
  const [frameInstance, setFrameInstance] = useState<PaymentFrame | null>(null);
  const [containerRef, setContainerRef] = useState<RefObject<HTMLDivElement> | null>(null);
  const [error, setError] = useState(false);
  const [width, setWidth] = useState<string | undefined>();
  const [height, setHeight] = useState<string | undefined>();

  const sdk = useMemo(() => {
    return TrustSDK.create({
      apiHost: ['rc', 'prod'].includes(yenv)
        ? 'https://trust.yandex.ru'
        : 'https://trust-test.yandex.ru',
    });
  }, [yenv]);

  const initBindCard = useCallback(
    async (containerRef: RefObject<HTMLDivElement>, url?: string | null) => {
      setContainerRef(containerRef);
      try {
        const frame = await sdk.bindCard(url ? { url } : undefined);

        setFrameInstance(frame);
      } catch (error) {
        setError(true);
      }
    },
    [sdk],
  );

  useEffect(() => {
    if (!frameInstance || !containerRef) {
      return;
    }

    frameInstance.on('success', () => {
      onSuccess();
      frameInstance.unmount();
    });

    frameInstance.on('error', () => setError(true));
    frameInstance.on('resize', (msg) => {
      setWidth(msg.size.width);
      setHeight(msg.size.height);
    });

    frameInstance.on('abort', onClose);

    try {
      // Монтируем фрейм в DOM
      frameInstance.mount('#' + containerRef.current?.id, {
        className: 'cardbindingform',
        title: 'bind-frame-class',
        width: frameWidth,
        height: frameHeight,
      });
    } catch (err) {
      setError(true);
    }
  }, [frameInstance, containerRef, onClose, onSuccess, frameWidth, frameHeight]);

  return { initBindCard, isError: error, width, height };
};
