import { VFC } from 'react';

import { RecoveryEmailList } from '@client/features/account-recovery-email';
import phoneImg from '@client/shared/assets/phone-filled.png';
import { Phone, PhoneShield, SocialProviders, Yandex } from '@client/shared/icons';
import { logProps } from '@client/shared/libs/metrika';
import { usePassportPaths } from '@client/shared/paths';
import { Hint, ImageIcon, List, ListItemLink, ListSkeleton, Section } from '@client/shared/ui-kit';

import { i18n } from './i18n';
import { useContactsData } from './api/store';
import additionalMailImg from './assets/additional-mail.png';
import defaultMailImg from './assets/default-mail.png';

const ContactsList = () => {
  const { isLoading, data, error } = useContactsData();
  const passportPaths = usePassportPaths();

  if (isLoading) {
    return <ListSkeleton count={5} />;
  }

  if (error) {
    return null;
  }

  const { defaultEmail, defaultPhone, additionalPhone, hasRecoveryEmail, externalAccounts } = data;

  return (
    <List>
      <ListItemLink
        href={passportPaths.emails}
        icon={<Yandex />}
        label={defaultEmail ? i18n('Email в Яндексе') : undefined}
        text={defaultEmail?.address ?? i18n('Создать email в Яндексе')}
        description={defaultEmail ? undefined : i18n('Ваш доступ ко всем сервисам')}
        {...logProps('listitem-link.edit-default-email')}
        // type={data.defaultEmail ? 'kebab' : 'add'}
      />
      <ListItemLink
        href={passportPaths.phones}
        icon={<PhoneShield />}
        label={defaultPhone ? i18n('Основной телефон') : undefined}
        text={defaultPhone?.number ?? i18n('Добавить телефон')}
        description={defaultPhone ? undefined : i18n('Поможет усилить защиту данных')}
        {...logProps('listitem-link.edit-default-phone')}
        // type={data.defaultPhone ? 'kebab' : 'add'}
      />
      {additionalPhone && (
        <ListItemLink
          href={passportPaths.phones}
          icon={<Phone />}
          label={i18n('Дополнительный телефон')}
          text={additionalPhone.number}
          {...logProps('listitem-link.edit-additional-phone')}
          // type="kebab"
        />
      )}

      {hasRecoveryEmail && <RecoveryEmailList />}

      {!externalAccounts.length && (
        <ListItemLink
          href={passportPaths.social}
          icon={<SocialProviders />}
          text={i18n('Добавить аккаунты в соцсетях')}
          description={i18n('Помогут быстрее входить в Яндекс и заполнить данные')}
          type="add"
          {...logProps('listitem-link.add-external-accounts')}
        />
      )}
      {Boolean(externalAccounts.length) && (
        <ListItemLink
          href={passportPaths.social}
          icon={<SocialProviders />}
          text={i18n('Связанные аккаунты')}
          description={i18n('ВК, ОК, Госуслуги, и т.д.')}
          counter={externalAccounts.length}
          {...logProps('listitem-link.edit-external-accounts')}
        />
      )}
    </List>
  );
};

export const Contacts: VFC = () => {
  const { data } = useContactsData();
  const passportPaths = usePassportPaths();

  const entriesForRequest = {
    additionalEmail: {
      text: i18n('запасную почту', {
        context: 'Название контакта, который просим добавить в плашке над секцией контактов',
      }),
      url: passportPaths.emailAdd,
      src: additionalMailImg.src,
    },
    defaultPhone: {
      text: i18n('телефон', {
        context: 'Название контакта, который просим добавить в плашке над секцией контактов',
      }),
      url: passportPaths.phones,
      src: phoneImg.src,
    },
    defaultEmail: {
      text: i18n('почту в Яндексе', {
        context: 'Название контакта, который просим добавить в плашке над секцией контактов',
      }),
      url: passportPaths.emails,
      src: defaultMailImg.src,
    },
  };

  const requestedEntry = data?.entryToRequest ? entriesForRequest[data.entryToRequest] : null;

  return (
    <Section>
      <Section.Title anchor="contacts">{i18n('Контакты')}</Section.Title>
      {requestedEntry && (
        <Hint
          text={i18n('Добавьте {contact}, чтобы открыть все возможности аккаунта Яндекса', {
            contact: requestedEntry.text,
          })}
          href={requestedEntry.url}
          before={<ImageIcon src={requestedEntry.src} size={44} />}
          variant="info"
        />
      )}
      <ContactsList />
    </Section>
  );
};
