import { useStore } from 'effector-react';
import { VFC } from 'react';

import {
  AddDocumentsFrequently,
  AddDocumentsList,
  DocumentEditor,
} from '@client/features/document-editor';
import { DocumentSection, useDocumentsThumbListQuery } from '@client/shared/api/graphql';
import { useDevice } from '@client/shared/hooks';
import { Plus } from '@client/shared/icons';
import { useEnvContext } from '@client/shared/libs/env';
import {
  AppBar,
  Button,
  Dialog,
  FloatContainer,
  Section,
  SectionStack,
} from '@client/shared/ui-kit';

import { i18n } from './i18n';
import { AddDocumentButton } from './components/AddDocumentButton';
import { Banner } from './components/Banner';
import { SafetyHint } from './components/SafetyHint';
import { ThumbList } from './components/ThumbList';
import * as model from './model';
import { ThumbItem } from './types';
import { splitDocumentsIntoSections } from './utils';

import styles from './Documents.module.css';

interface SectionProps {
  splitedDocuments: {
    section: DocumentSection;
    documents: ThumbItem[];
  }[];
}

const ThumbsSection: VFC<SectionProps> = (props) => {
  const { splitedDocuments } = props;
  const {
    ua: { isSearchApp, isYandexApp },
  } = useEnvContext();

  return (
    <Section className={styles.section}>
      <div className={styles.thumbs}>
        {splitedDocuments.map((documents) => (
          <ThumbList
            key={documents.section}
            documents={documents.documents}
            section={documents.section}
          />
        ))}
      </div>
      {isYandexApp && isSearchApp && splitedDocuments.length >= 2 && <SafetyHint />}
    </Section>
  );
};

const EmptyDocumentsSection: VFC = () => {
  const {
    ua: { isSearchApp, isYandexApp },
  } = useEnvContext();

  return (
    <Section className={styles.section}>
      <Banner />
      <AddDocumentsFrequently />
      <AddDocumentsList />
      {isYandexApp && isSearchApp && (
        <div className={styles.hint}>
          <SafetyHint />
        </div>
      )}
    </Section>
  );
};

const AddDocumentsSection: VFC<SectionProps> = (props) => {
  const { splitedDocuments } = props;
  const { isTouch } = useDevice();
  const {
    ua: { isYandexApp, isSearchApp },
  } = useEnvContext();

  if (splitedDocuments.length >= 2 && isTouch) {
    return (
      <FloatContainer>
        <Button shape="circle" variant="action" size="xl" onPress={model.addDocumentsDialog.show}>
          <Plus />
        </Button>
      </FloatContainer>
    );
  }

  if (splitedDocuments.length >= 2 && !isTouch) {
    return (
      <Section className={styles.section}>
        <AddDocumentButton onPress={model.addDocumentsDialog.show} />
      </Section>
    );
  }

  return (
    <Section className={styles.section}>
      <AddDocumentsFrequently />
      <AddDocumentsList />
      {isYandexApp && isSearchApp && (
        <div className={styles.hint}>
          <SafetyHint />
        </div>
      )}
    </Section>
  );
};

const DocumentsBody: VFC = () => {
  const { data, loading: isLoading, error } = useDocumentsThumbListQuery();

  const isAddDocumentsDialogVisible = useStore(model.addDocumentsDialog.$isVisible);

  if (isLoading || !data || error) {
    return null;
  }

  const splitedDocuments = splitDocumentsIntoSections(data?.documents || []);

  if (splitedDocuments.length === 0) {
    return (
      <SectionStack>
        <EmptyDocumentsSection />
      </SectionStack>
    );
  }

  return (
    <>
      <SectionStack>
        <ThumbsSection splitedDocuments={splitedDocuments} />
        <AddDocumentsSection splitedDocuments={splitedDocuments} />
      </SectionStack>
      <Dialog
        className={styles.dialog}
        hasClose
        visible={isAddDocumentsDialogVisible}
        onClose={model.addDocumentsDialog.hide}
      >
        <SectionStack className={styles.sectionStack}>
          <Section className={styles.section}>
            <AddDocumentsFrequently onTilePress={model.addDocumentsDialog.hide} />
          </Section>
          <Section className={styles.section}>
            <AddDocumentsList onItemPress={model.addDocumentsDialog.hide} />
          </Section>
        </SectionStack>
      </Dialog>
    </>
  );
};

export const Documents: VFC = () => {
  const {
    ua: { isYandexApp, isSearchApp },
  } = useEnvContext();

  return (
    <>
      {isSearchApp && isYandexApp && <AppBar title={i18n('Документы')} />}
      <DocumentsBody />
      <DocumentEditor />
    </>
  );
};
