import { VFC } from 'react';

import { InsecureAuthHint } from '@client/entities/InsecureAuthHint';
import { AccountType, AuthType, useEnterMethodsQuery } from '@client/shared/api/graphql';
import { KeyRefresh, Login, Phone } from '@client/shared/icons';
import { YandexKey } from '@client/shared/icons/colored';
import { logProps } from '@client/shared/libs/metrika';
import { useIdPaths, usePassportPaths } from '@client/shared/paths';
import { List, ListItemLink, ListSkeleton, Section } from '@client/shared/ui-kit';

import { i18n } from './i18n';
import { getAuthTypeText, getPasswordUpdateDescription } from './utils';

const EnterMethodsList = () => {
  const { loading, data, error } = useEnterMethodsQuery();
  const idPaths = useIdPaths();
  const passportPaths = usePassportPaths();

  if (error) {
    return null;
  }

  if (loading || !data) {
    return <ListSkeleton count={2} />;
  }

  const { type, phones, passwordInfo = null, authType = null } = data.account;
  const primaryPhone = phones.find((phone) => phone.isPrimary);
  const showPhoneForSms = authType === AuthType.SmsOnly || authType === AuthType.PasswordAndSms;
  const showPasswordRefresh =
    authType === AuthType.PasswordOnly || authType === AuthType.PasswordAndSms;

  return (
    <>
      {authType === null && (
        <InsecureAuthHint
          type={type === AccountType.Social ? 'external-social' : 'external-email'}
        />
      )}

      <List>
        <ListItemLink
          {...logProps('listitem.login-methods')}
          href={idPaths.securityEnterMethods}
          icon={<Login />}
          text={getAuthTypeText(authType)}
        />

        {showPhoneForSms && primaryPhone && (
          <ListItemLink
            {...logProps('listitem.phones')}
            href={passportPaths.phones}
            icon={<Phone />}
            text={i18n('Телефон для смс')}
            description={primaryPhone.number}
          />
        )}

        {showPasswordRefresh && passwordInfo && (
          <ListItemLink
            {...logProps('listitem.set-password')}
            href={passportPaths.changePassword}
            icon={<KeyRefresh />}
            text={i18n('Обновить пароль')}
            description={getPasswordUpdateDescription(passwordInfo.updatedAt)}
          />
        )}

        {authType === AuthType.YandexKey && (
          <ListItemLink
            {...logProps('listitem.set-yandex-key')}
            href={passportPaths.accesses2fa}
            icon={<YandexKey />}
            text={i18n('Настроить Ключ')}
          />
        )}
      </List>
    </>
  );
};

export const EnterMethods: VFC = () => {
  return (
    <Section>
      <Section.Title anchor="enter-methods">{i18n('Способ входа')}</Section.Title>
      <Section.Description>{i18n('Как вы заходите на сервисы Яндекса')}</Section.Description>

      <EnterMethodsList />
    </Section>
  );
};
