import { MouseEventHandler, VFC, useCallback, useMemo, useState } from 'react';

import { useFamilyMembersWidgetQuery } from '@client/shared/api/graphql';
import { Add } from '@client/shared/icons';
import { getAvatarUrl, useAvatarsPath } from '@client/shared/paths';
import { Dialog, ListItem, Section } from '@client/shared/ui-kit';

import { MemberData } from './types';
import { FamilyMember, FamilyMemberSkeleton } from './ui/FamilyMember';
import { MemberCard } from './ui/MemberCard';

import styles from './FamilyMembers.module.css';

interface FamilyListProps {
  avatarHost: string;
  items: MemberData[];
  withInviteButton: boolean;
  onItemClick: MouseEventHandler<HTMLDivElement>;
}

export const FamilyList: VFC<FamilyListProps> = ({ items, withInviteButton, onItemClick }) => {
  return (
    <>
      {items.map((item) => (
        <FamilyMember key={item.id} data={item} onClick={onItemClick} />
      ))}

      {withInviteButton && (
        <ListItem className={styles.add} tabIndex={0} icon={<Add />} text="Добавить участника" />
      )}
    </>
  );
};

export const FamilyMembersWidget = () => {
  const { data, error, loading } = useFamilyMembersWidgetQuery();
  const [drawerUser, setDrawerUser] = useState('');
  const avatarHost = useAvatarsPath();

  const closeMemberCard = useCallback(() => setDrawerUser(''), []);
  const onItemClick: MouseEventHandler<HTMLDivElement> = useCallback((e) => {
    setDrawerUser(e.currentTarget.dataset.id || '');
  }, []);

  const items = useMemo(() => {
    if (!data) return [];

    const { family, viewer } = data;
    const items: MemberData[] = [];

    if (family.info?.id) {
      if (family.members) {
        for (const member of family.members) {
          items.push({
            id: member.uid,
            avatar: getAvatarUrl(avatarHost, member.avatar || ''),
            type: member.uid === family.info.adminUid ? 'admin' : 'member',
            name: member.name,
            hasSecurePhone: Boolean(member.hasSecurePhone),
          });
        }
      }

      if (family.invites) {
        for (const invite of family.invites) {
          items.push({
            id: invite.id,
            avatar: getAvatarUrl(avatarHost, ''),
            name: invite.contact,
            type: 'invite',
          });
        }
      }

      if (family.kids) {
        for (const kid of family.kids) {
          items.push({
            id: kid.uid,
            avatar: getAvatarUrl(avatarHost, kid.avatar || ''),
            name: kid.name,
            type: 'child',
          });
        }
      }
    } else {
      items.push({
        id: String(viewer.id),
        avatar: getAvatarUrl(avatarHost, viewer.avatarId || ''),
        type: 'admin',
        name: viewer.displayName,
      });
    }

    return items;
  }, [data, avatarHost]);

  if (loading) {
    return (
      <Section className={styles.members}>
        <FamilyMemberSkeleton key="1" />
        <FamilyMemberSkeleton key="2" />
        <FamilyMemberSkeleton key="3" />
      </Section>
    );
  }

  if (error || !data) {
    return null;
  }

  const hasFamily = Boolean(data.family.info?.id);
  const member = items.find((item) => item.id === drawerUser);
  const memberPayInfo = data.family.pay?.userRestrictions.find((item) => item.uid === drawerUser);

  return (
    <Section className={styles.members}>
      <Dialog
        className={styles.dialog}
        visible={Boolean(drawerUser && member)}
        onClose={closeMemberCard}
        hasClose={false}
      >
        {member && (
          <MemberCard
            member={member}
            payData={memberPayInfo}
            isAdminView={data.viewer.id === data.family.info?.adminUid}
            isSelfView={member.id === data.viewer.id}
            onClose={closeMemberCard}
          />
        )}
      </Dialog>

      <FamilyList
        avatarHost={avatarHost}
        items={items}
        withInviteButton={hasFamily ? data.family.info?.adminUid === data.viewer.id : true}
        onItemClick={onItemClick}
      />
    </Section>
  );
};
