import { VFC } from 'react';

import alertSrc from '@client/shared/assets/alert.png';
import {
  ArrowShortForward,
  Ban,
  FamilyCardError,
  FamilyCardLocked,
  FamilyCardOutline,
  HeartBroken,
} from '@client/shared/icons';
import {
  Button,
  CardTemplate,
  Hint,
  ImageIcon,
  List,
  ListItem,
  Section,
  SectionStack,
  Text,
} from '@client/shared/ui-kit';
import { formatCurrency } from '@shared/helpers';

import { MemberData } from '../../types';
import { TEXTS } from './constants';

import styles from './MemberCard.module.css';

interface PayData {
  enabled?: boolean | null;
  uid: string;
  unlim?: boolean | null;
  limitCurrency: string;
  limit: { value: number; limitMode: string };
  balance: number;
}

interface MemberCardProps {
  isAdminView: boolean;
  isSelfView: boolean;

  member: MemberData;
  payData?: PayData | null;

  onClose: () => void;
}

const getFamilyPayDescription = (member: MemberData, payData: PayData) => {
  if (!payData.enabled) {
    return TEXTS.ACCESS_RESTRICTED;
  }

  if (!member.hasSecurePhone) {
    return TEXTS.NO_ACCESS;
  }

  if (payData.unlim) {
    return TEXTS.UNLIM;
  }

  return `${payData.balance}${formatCurrency(payData.limitCurrency)} ${
    TEXTS.LIMIT_PERIODS[payData.limit.limitMode]
  }`;
};

const getFamilyPayIcon = (hasPhone?: boolean, familyCardEnabled?: boolean) => {
  if (!familyCardEnabled) return <FamilyCardLocked />;
  if (!hasPhone) return <FamilyCardError className={styles.alert} />;

  return <FamilyCardOutline />;
};

export const MemberCard: VFC<MemberCardProps> = ({
  member,
  payData,

  isAdminView,
  isSelfView,

  onClose,
}) => {
  const isInvite = member.type === 'invite';
  const familyCardEnabled = Boolean(payData?.enabled);
  const hasSecurePhone = member.hasSecurePhone;

  const showAccountActions = !isInvite && (isAdminView || isSelfView);
  const hasPhoneAlert = familyCardEnabled && !hasSecurePhone;

  const title = isInvite ? TEXTS.INVITE_TITLE : member.name;
  const description = isInvite ? TEXTS.INVITE_DESCRIPTION : TEXTS.ROLES[member.type];

  const FamilyIcon = getFamilyPayIcon(hasSecurePhone, familyCardEnabled);

  return (
    <div className={styles.root}>
      <CardTemplate>
        <CardTemplate.Content>
          <SectionStack className={styles.content}>
            <Section className={styles.section}>
              <Section.Media>
                {isInvite ? (
                  <div className={styles.clock} />
                ) : (
                  <img className={styles.avatar} src={member.avatar} />
                )}
              </Section.Media>

              <div className={styles.header}>
                <div className={styles.name}>
                  <Text variant="heading-l" weight="bold">
                    {title}
                  </Text>
                </div>

                <div className={styles.role}>
                  <Text variant="text-m-long" color="secondary">
                    {description}
                  </Text>
                </div>
              </div>

              {showAccountActions && hasPhoneAlert && (
                <Hint
                  variant="negative"
                  text={TEXTS.PHONE_ALERT(isAdminView)}
                  before={<ImageIcon src={alertSrc} size={44} />}
                />
              )}
            </Section>

            {showAccountActions && (
              <Section className={styles.section}>
                <List>
                  {!(isAdminView && isSelfView) && payData && (
                    <ListItem
                      icon={FamilyIcon}
                      href="/"
                      after={<ArrowShortForward />}
                      text={TEXTS.FAMILY_PAY}
                      description={getFamilyPayDescription(member, payData)}
                      plateColor={hasPhoneAlert ? 'red-gradient100' : 'default'}
                    />
                  )}
                  {isAdminView && isSelfView && (
                    <ListItem icon={<HeartBroken />} text={TEXTS.DELETE_ACCOUNT} />
                  )}
                  {!isAdminView && isSelfView && (
                    <ListItem icon={<HeartBroken />} text={TEXTS.QUIT_FAMILY} />
                  )}
                  {isAdminView && !isSelfView && <ListItem icon={<Ban />} text={TEXTS.KICK_USER} />}
                </List>
              </Section>
            )}
          </SectionStack>
        </CardTemplate.Content>
        <CardTemplate.Actions>
          <div className={styles.buttons}>
            {isInvite ? (
              <>
                <Button variant="default" size="xl">
                  {TEXTS.RESEND_INVITE}
                </Button>
                <Button variant="clear" size="xl" onPress={onClose}>
                  {TEXTS.CANCEL_INVITE}
                </Button>
              </>
            ) : (
              <Button variant="default" size="xl" onPress={onClose}>
                {TEXTS.CLOSE}
              </Button>
            )}
          </div>
        </CardTemplate.Actions>
      </CardTemplate>
    </div>
  );
};
