import { intervalToDuration } from 'date-fns';
import { useEvent } from 'effector-react';
import { FC } from 'react';

import { SplitDialog, SplitModel } from '@client/features/split';
import { FuturePayment, FuturePaymentType } from '@client/shared/api/graphql';
import { ListItemView, Text } from '@client/shared/ui-kit';
import { formatCurrency } from '@shared/helpers';

import marketLogoImage from './assets/market-logo.png';
import splitLogoImage from './assets/split-logo.png';
import { i18n } from './i18n';

import styles from './FuturePaymentItem.module.css';

export interface FuturePaymentItemProps {
  payment: FuturePayment;
}

interface ServiceInfo {
  image: string;
  title: () => string;
}

const servicesMap: Record<string, ServiceInfo> = {
  market: {
    image: marketLogoImage.src,
    title: () => i18n('Маркет'),
  },
};

export const FuturePaymentItem: FC<FuturePaymentItemProps> = (props) => {
  const { payment } = props;
  const showDialog = useEvent(SplitModel.show); // TODO: нужно будет, наверное, возвращать разные колбэки в зависимости от типа платежа

  const interval = intervalToDuration({
    start: new Date(),
    end: new Date(payment.nextPaymentAt),
  });

  const estimate = interval.months
    ? i18n('через {count} месяцев', { count: interval.months })
    : i18n('через {count} дней', { count: interval.days ?? 0 });

  return (
    <>
      <ListItemView className={styles.root} onPress={showDialog}>
        <ListItemView.Before>
          <div className={styles.serviceImageWrap}>
            <img src={servicesMap[payment.service].image} className={styles.serviceImage} />
            {payment.type === FuturePaymentType.Split && (
              <img src={splitLogoImage.src} className={styles.typeImage} />
            )}
          </div>
        </ListItemView.Before>
        <ListItemView.Content>
          <Text variant="text-m" weight="medium">
            {servicesMap[payment.service].title()}
          </Text>
          {payment.type === FuturePaymentType.Split && (
            <Text variant="text-s" color="secondary">
              {i18n('Сплит')}
            </Text>
          )}
        </ListItemView.Content>
        <ListItemView.After>
          <div className={styles.afterSlotContent}>
            <Text variant="text-m" weight="medium">
              {payment.amount} {formatCurrency('RUB')}
            </Text>
            <Text variant="text-s">{estimate}</Text>
          </div>
        </ListItemView.After>
      </ListItemView>
      {payment.iframeUrl && <SplitDialog url={payment.iframeUrl} />}
    </>
  );
};
