import { useRouter } from 'next/router';
import { VFC, useCallback, useState } from 'react';

import { addGibddDocument } from '@client/features/gibdd/api/store';
import { BillsDocumentType, useGibddDataQuery } from '@client/shared/api/graphql';
import { useForm } from '@client/shared/libs/effector-forms';
import { Button, Section, Text, TextField } from '@client/shared/ui-kit';

import { GIBDD_DOCUMENTS_MASK, TEXTS, getErrorText, useGibddPaths } from '../../libs';
import * as model from '../../model';
import { GibddHeader } from '../GibddHeader';
import { i18n } from './i18n';

interface GibddAddDocumentProps {
  type: BillsDocumentType;
}

interface GibddAddForm extends GibddAddDocumentProps {
  onAdded(): void;
}

const GibddDocumentAddForm: VFC<GibddAddForm> = ({ type, onAdded }) => {
  const { fields } = useForm(model.setDocumentNumberForm());
  const [isLoading, setLoading] = useState(false);

  const field = fields[type];

  const onFieldBlur = useCallback(() => {
    field.onBlur();
  }, [field]);

  const onSave = useCallback(() => {
    setLoading(true);

    addGibddDocument({ value: field.value, title: TEXTS[type].title(), type })
      .then(() => {
        onAdded();
      })
      .catch(() => {
        setLoading(false);
      });
  }, [type, field, onAdded]);

  return (
    <>
      <Text variant="heading-l">{TEXTS[type].heading()}</Text>
      <TextField
        label={TEXTS[type].label()}
        value={field.value}
        placeholder={TEXTS[type].placeholder}
        onChangeValue={field.onChange}
        mask={GIBDD_DOCUMENTS_MASK}
        error={!field.isValid ? getErrorText(type) : undefined}
        onBlur={onFieldBlur}
      />
      <Button variant="action" size="xl" disabled={!field.isValid || isLoading} onPress={onSave}>
        {i18n('Добавить и искать штрафы')}
      </Button>
    </>
  );
};

export const GibddAddDocument: VFC<GibddAddDocumentProps> = ({ type }) => {
  const { refetch } = useGibddDataQuery();
  const router = useRouter();
  const { gibdd, gibddSettings } = useGibddPaths();

  const onAdded = useCallback(() => {
    refetch().then(() => {
      router.push(gibdd);
    });
  }, [refetch, router, gibdd]);

  return (
    <Section>
      <GibddHeader backPath={gibddSettings} />
      <GibddDocumentAddForm type={type} onAdded={onAdded} />
    </Section>
  );
};
