import cx from 'classnames';
import { VFC, useMemo } from 'react';

import { isExpired } from '@client/features/gibdd';
import { GibddDataQuery } from '@client/shared/api/graphql';
import { Clock, Close, Error } from '@client/shared/icons';
import { Button, Dialog, Text } from '@client/shared/ui-kit';
import { formatPrice } from '@shared/helpers';
import { useDateFormatter } from '@use-platform/react';
import { useTheme } from '@yandex-id/design-system';

import { getDocumentName } from '../../libs';
import { YandexPayButton } from '../YandexPayButton';
import { i18n } from './i18n';

import styles from './GibddBillInfo.module.css';

type Bill = GibddDataQuery['bills']['bills'][0];
type Document = GibddDataQuery['billsDocuments'][0];

interface GibddBillInfoProps {
  data: Bill;
  document?: Document;
  visible: boolean;
  onClose(): void;
}

export const GibddBillInfo: VFC<GibddBillInfoProps> = ({ data, visible, onClose, document }) => {
  const { themeProps } = useTheme({ theme: 'default', colorScheme: 'dark' });
  const formatDate = useDateFormatter({ dateStyle: 'short' });

  const isExpiredBill = isExpired(data);

  const resolution = useMemo(
    () =>
      [
        `${i18n('Постановление №')}${data.supplier_bill_id}`,
        `${i18n('от')} ${formatDate.format(Date.parse(data.bill_date))}`,
        document &&
          `${i18n('выписан по документу')} ${getDocumentName(document?.type)} ${document.value}`,
      ]
        .filter(Boolean)
        .join(' '),
    [data.supplier_bill_id, data.bill_date, formatDate, document],
  );

  const details = useMemo(() => {
    const arr = [];

    if (data.purpose || data.legal_act) {
      if (data.purpose && data.legal_act) {
        arr.push({
          name: i18n('Назначение платежа'),
          content: `${data.purpose}, ${data.legal_act}`,
        });
      } else if (data.purpose) {
        arr.push({
          name: i18n('Назначение платежа'),
          content: `${data.purpose}`,
        });
      } else {
        arr.push({
          name: i18n('Назначение платежа'),
          content: `${data.legal_act}`,
        });
      }
    }

    if (data.offense_place) {
      arr.push({
        name: i18n('Место нарушения'),
        content: `${data.offense_place}`,
      });
    }

    if (data.payer_name) {
      arr.push({
        name: i18n('Плательщик'),
        content: `${data.payer_name}`,
      });
    }

    if (data.payee_name) {
      arr.push({
        name: i18n('Получатель платежа'),
        content: `${data.payee_name}`,
      });
    }

    if (data.account_number) {
      arr.push({
        name: i18n('Р/счёт'),
        content: `${data.account_number}`,
      });
    }

    if (data.inn) {
      arr.push({
        name: i18n('ИНН'),
        content: `${data.inn}`,
      });
    }

    if (data.kpp) {
      arr.push({
        name: i18n('КПП'),
        content: `${data.kpp}`,
      });
    }

    if (data.bik) {
      arr.push({
        name: i18n('БИК'),
        content: `${data.bik}`,
      });
    }

    if (data.oktmo) {
      arr.push({
        name: i18n('ОКТМО'),
        content: `${data.oktmo}`,
      });
    }

    return arr;
  }, [data]);

  return (
    <Dialog className={styles.root} visible={visible} onClose={onClose} hasClose={false}>
      <section className={styles.header} {...themeProps}>
        <div className={data.payment_deadline} />
        <div className={styles.icon} />
        <Text variant="heading-m" as="h4">
          {i18n('Штраф ГИБДД')}
        </Text>
        <Text className={styles.billNumber} variant="text-s">
          № {data.supplier_bill_id}
        </Text>
        <Text className={styles.amount} variant="display-s">
          {formatPrice(data.discounted_amount, 'RUB', false, { signed: false })}
        </Text>

        <div className={styles.status}>
          {isExpiredBill ? (
            <Error className={cx(styles.statusIcon, styles.expiredIcon)} />
          ) : (
            <Clock className={styles.statusIcon} />
          )}
          <Text variant="text-s">
            {isExpiredBill
              ? i18n('Просрочен')
              : i18n('До {date}', { date: formatDate.format(new Date(data.payment_deadline)) })}
          </Text>
        </div>

        <Button
          className={styles.close}
          variant="clear"
          size="l"
          onPress={onClose}
          aria-label={i18n('Закрыть')}
        >
          <Close />
        </Button>
      </section>
      <section className={styles.body}>
        <Text variant="text-m" color="secondary">
          {i18n('Детали штрафа')}
        </Text>
        <Text className={styles.resolution} variant="text-m">
          {resolution}
        </Text>
        {details.map(({ name, content }) => (
          <>
            <Text className={styles.detailsName} variant="text-s" color="secondary">
              {name}
            </Text>
            <Text className={styles.detailsContent} variant="text-m">
              {content}
            </Text>
          </>
        ))}
      </section>
      <YandexPayButton bills={[data]} />
    </Dialog>
  );
};
