import { VFC, useCallback, useState } from 'react';

import { GibddBill, isExpired } from '@client/features/gibdd';
import { restartBillsSearch } from '@client/features/gibdd/api/store';
import { GibddDataQuery, useBillsDataQuery } from '@client/shared/api/graphql';
import { Good, Sync } from '@client/shared/icons';
import { Button, Link, PromoContent, Text } from '@client/shared/ui-kit';

import { GibddBillInfo } from '../GibddBillInfo';
import { GibddPaymentStatus } from '../GibddPaymentStatus';
import { YandexPayButton } from '../YandexPayButton';
import { GibddBillCard, GibddBillHint } from './components';
import { i18n } from './i18n';

import styles from './GibddBills.module.css';

interface GibddBillsProps {
  bills: GibddBill[];
  documents: GibddDataQuery['billsDocuments'];
}

export const GibddBills: VFC<GibddBillsProps> = ({ bills, documents }) => {
  const [visible, setVisible] = useState(false);
  const [selected, setSelected] = useState<GibddBill | null>(null);
  const [selectedBills, setSelectedBills] = useState<GibddBill[]>(bills);

  const { refetch } = useBillsDataQuery();
  const onOpenBill = useCallback(
    (data) => () => {
      setSelected(data);
      setVisible(true);
    },
    [],
  );

  const onCloseBill = useCallback(() => {
    setVisible(false);
  }, []);

  const onRestartSearch = useCallback(
    (e) => {
      e?.preventDefault && e.preventDefault();

      restartBillsSearch().then(() => {
        refetch();
      });
    },
    [refetch],
  );

  const onSelecetBill = useCallback(
    (bill: GibddBill) => () => {
      const isSelected = selectedBills.find(({ bill_id }) => bill.bill_id === bill_id);

      if (isSelected) {
        setSelectedBills(selectedBills.filter(({ bill_id }) => bill.bill_id !== bill_id));
      } else {
        setSelectedBills([...selectedBills, bill]);
      }
    },
    [selectedBills],
  );

  const hint = <GibddBillHint documents={documents} />;

  if (bills.length === 0) {
    return (
      <>
        <PromoContent className={styles.promo}>
          <PromoContent.Image>
            <Good />
          </PromoContent.Image>
          <PromoContent.Title>{i18n('Штрафов не найдено, так держать!')}</PromoContent.Title>
          <PromoContent.Button>
            <Button className={styles.button} variant="default" size="xl" onPress={onRestartSearch}>
              {i18n('Искать снова')}
            </Button>
          </PromoContent.Button>
        </PromoContent>
        {hint}
      </>
    );
  }

  return (
    <>
      <div className={styles.heading}>
        <Text variant="heading-l">{i18n('Найдено {count} штрафов', { count: bills.length })}</Text>
        <Link href="#" className={styles.restart} size="s" onClick={onRestartSearch}>
          <Sync width={16} height={16} />
          {i18n('Искать снова')}
        </Link>
      </div>

      <div className={styles.bills}>
        {Array.from(bills)
          .sort((a, b) => Date.parse(a.payment_deadline) - Date.parse(b.payment_deadline))
          .map((data) => (
            <GibddBillCard
              key={data.bill_id}
              amount={data.amount}
              discount={data.discounted_amount}
              isExpired={isExpired(data)}
              deadline={data.payment_deadline}
              onPress={onOpenBill(data)}
              onSelect={onSelecetBill(data)}
              selected={Boolean(selectedBills.find(({ bill_id }) => bill_id === data.bill_id))}
              document={documents.find((doc) => doc.document_id === data.document_id)}
            />
          ))}
      </div>

      {hint}

      <GibddPaymentStatus />

      <YandexPayButton bills={selectedBills} sticky />

      {selected && (
        <GibddBillInfo
          visible={visible}
          data={selected}
          onClose={onCloseBill}
          document={documents.find((doc) => doc.document_id === selected.document_id)}
        />
      )}
    </>
  );
};
