import cx from 'classnames';
import { VFC } from 'react';

import { GibddDataQuery } from '@client/shared/api/graphql';
import { Checkbox, Text } from '@client/shared/ui-kit';
import { formatPrice } from '@shared/helpers';
import { PressProps, useDateFormatter, usePress } from '@use-platform/react';

import { i18n } from './i18n';

import styles from './GibddBillCard.module.css';

interface GibddBillCardProps extends PressProps {
  document?: GibddDataQuery['billsDocuments'][0];
  deadline: string;
  amount: number;
  isExpired: boolean;
  discount: number;
  onSelect(): void;
  selected?: boolean;
}

export const GibddBillCard: VFC<GibddBillCardProps> = ({
  document,
  deadline,
  amount,
  discount,
  isExpired,
  onSelect,
  selected,
  ...otherProps
}) => {
  const formatDate = useDateFormatter({ dateStyle: 'short' });
  const { pressProps } = usePress(otherProps);

  return (
    <div
      className={cx(styles.root, { [styles.isExpired]: isExpired })}
      tabIndex={0}
      {...pressProps}
    >
      <div className={styles.price}>
        <Text className={styles.amount} variant="heading-m">
          {formatPrice(discount, 'RUB', false, { signed: false })}
        </Text>
        {discount !== amount && (
          <Text className={styles.origAmount} variant="text-m">
            {formatPrice(amount, 'RUB', false, { signed: false })}
          </Text>
        )}
      </div>

      {document && (
        <Text className={styles.document} variant="text-s">
          {document.title} {document.value}
        </Text>
      )}
      <Text className={styles.date} variant="text-s" color={isExpired ? 'negative' : 'secondary'}>
        {isExpired
          ? i18n('Просрочен')
          : i18n('До {date}', { date: formatDate.format(new Date(deadline)) })}
      </Text>
      <Checkbox
        className={styles.checkbox}
        size="s"
        variant="outline"
        checked={selected}
        onChangeValue={onSelect}
      />
    </div>
  );
};
