import { useRouter } from 'next/router';
import { VFC, useCallback, useState } from 'react';

import { deleteGibddDocument, updateGibddDocument } from '@client/features/gibdd/api/store';
import { BillsDocumentType, GibddDataQuery, useGibddDataQuery } from '@client/shared/api/graphql';
import { useForm } from '@client/shared/libs/effector-forms';
import { Button, Section, Skeleton, Text, TextField } from '@client/shared/ui-kit';

import { getErrorText, useGibddPaths } from '../../libs';
import * as model from '../../model';
import { GibddHeader } from '../GibddHeader';
import { i18n } from './i18n';

type Document = GibddDataQuery['billsDocuments'][0];

interface FormProps {
  document: Document;
  onEdit(): void;
}

type TextsType = {
  heading: () => string;
  label: () => string;
};

const TEXTS: Record<BillsDocumentType, TextsType> = {
  [BillsDocumentType.DriverLicense]: {
    heading: () => i18n('Редактировать ВУ'),
    label: () => i18n('Номер ВУ'),
  },
  [BillsDocumentType.VehicleRegistrationCertificate]: {
    heading: () => i18n('Редактировать СТС'),
    label: () => i18n('Номер СТС'),
  },
};

const GibddDocumentEditForm: VFC<FormProps> = ({
  document: { document_id, value, title, type },
  onEdit,
}) => {
  const { fields } = useForm(model.setDocumentNumberForm(value));
  const [isLoading, setLoading] = useState(false);

  const field = fields[type];

  const onFieldBlur = useCallback(() => {
    field.onBlur();
  }, [field]);

  const onSave = useCallback(() => {
    setLoading(true);

    updateGibddDocument({ id: document_id, value: field.value, title, type })
      .then(() => {
        onEdit();
      })
      .catch(() => {
        setLoading(false);
      });
  }, [document_id, title, type, field, onEdit]);

  const onDelete = useCallback(() => {
    setLoading(true);

    deleteGibddDocument({ id: document_id })
      .then(() => {
        onEdit();
      })
      .catch(() => {
        setLoading(false);
      });
  }, [document_id, onEdit]);

  return (
    <>
      <Text variant="heading-l">{TEXTS[type].heading()}</Text>
      <TextField
        label={TEXTS[type].label()}
        value={field.value}
        onChangeValue={field.onChange}
        error={!field.isValid ? getErrorText(type) : undefined}
        onBlur={onFieldBlur}
      />
      <Button variant="action" size="xl" disabled={!field.isValid || isLoading} onPress={onSave}>
        {i18n('Сохранить и искать штрафы')}
      </Button>
      <Button variant="clear" size="l" disabled={isLoading} onPress={onDelete}>
        {i18n('Удалить')}
      </Button>
    </>
  );
};

export const GibddDocument: VFC = () => {
  const router = useRouter();
  const { gibdd, gibddSettings } = useGibddPaths();

  const { loading, data, error, refetch } = useGibddDataQuery();

  const onEdit = useCallback(() => {
    refetch().then(() => {
      router.push(gibdd);
    });
  }, [router, refetch, gibdd]);

  if (loading) {
    return (
      <Skeleton height="240" width="100%">
        <rect width="100%" height="60" rx="24" />
        <rect y="80" width="100%" height="60" rx="24" />
        <rect y="160" width="100%" height="60" rx="24" />
      </Skeleton>
    );
  }

  if (!data || error) {
    return null;
  }

  const document = data.billsDocuments.find(({ document_id }) => document_id === router.query.id);

  if (!document) {
    return null;
  }

  return (
    <Section>
      <GibddHeader backPath={gibddSettings} />
      <GibddDocumentEditForm document={document} onEdit={onEdit} />
    </Section>
  );
};
