import { VFC, useCallback, useState } from 'react';

import { addGibddDocument } from '@client/features/gibdd/api/store';
import { BillsDocumentType } from '@client/shared/api/graphql';
import { useForm } from '@client/shared/libs/effector-forms';
import { Button, Link, Text, TextField } from '@client/shared/ui-kit';

import { GIBDD_DOCUMENTS_MASK, TEXTS, getErrorText } from '../../libs';
import * as model from '../../model';
import { i18n, i18nRaw } from './i18n';

import styles from './GibddOnboarding.module.css';

const LICENSE_AGREEMENT_LINK = 'https://yandex.ru/legal/search_mobile_agreement/';
const PRIVACY_POLICY_LINK = 'https://yandex.ru/legal/rules/';

interface GibddOnboardingProps {
  onComplete(): void;
}

export const GibddOnboarding: VFC<GibddOnboardingProps> = ({ onComplete }) => {
  const { fields, isValid, isDirty } = useForm(model.setGibddForm());
  const [isLoading, setLoading] = useState(false);

  const onCarFieldBlur = useCallback(() => {
    fields.car.onBlur();
  }, [fields.car]);

  const onDriverLicenseFieldBlur = useCallback(() => {
    fields.driverLicense.onBlur();
  }, [fields.driverLicense]);

  const onAddDocument = useCallback(() => {
    const promises = [];

    setLoading(true);

    if (fields.car.value && fields.car.isValid) {
      promises.push(
        addGibddDocument({
          type: BillsDocumentType.VehicleRegistrationCertificate,
          value: fields.car.value.replace(/ /g, ''),
          title: i18n('СТС'),
        }),
      );
    }

    if (fields.driverLicense.value && fields.driverLicense.isValid) {
      promises.push(
        addGibddDocument({
          type: BillsDocumentType.DriverLicense,
          value: fields.driverLicense.value.replace(/ /g, ''),
          title: i18n('ВУ'),
        }),
      );
    }

    Promise.all(promises).then(onComplete);
  }, [fields, onComplete]);

  return (
    <>
      <Text variant="heading-l">{i18n('Чтобы оплатить штрафы, добавьте номера СТС и ВУ')}</Text>

      <TextField
        label={TEXTS.VEHICLE_REGISTRATION_CERTIFICATE.label(true)}
        onChangeValue={fields.car.onChange}
        placeholder={TEXTS.VEHICLE_REGISTRATION_CERTIFICATE.placeholder}
        onBlur={onCarFieldBlur}
        value={fields.car.value}
        mask={GIBDD_DOCUMENTS_MASK}
        error={
          !fields.car.isValid
            ? getErrorText(BillsDocumentType.VehicleRegistrationCertificate)
            : undefined
        }
      />

      <TextField
        label={TEXTS.DRIVER_LICENSE.label(true)}
        onChangeValue={fields.driverLicense.onChange}
        placeholder={TEXTS.DRIVER_LICENSE.placeholder}
        onBlur={onDriverLicenseFieldBlur}
        value={fields.driverLicense.value}
        mask={GIBDD_DOCUMENTS_MASK}
        error={
          !fields.driverLicense.isValid ? getErrorText(BillsDocumentType.DriverLicense) : undefined
        }
      />

      <Button
        variant="action"
        size="xl"
        type="submit"
        onPress={onAddDocument}
        disabled={!isDirty || !isValid || isLoading}
      >
        {i18n('Искать штрафы')}
      </Button>

      <Text className={styles.agreement} variant="text-xs">
        {i18nRaw(
          'Нажимая на кнопку, я соглашаюсь с условиями {licenseAgreement} и {privacyPolicy}',
          {
            licenseAgreement: (
              <Link key={LICENSE_AGREEMENT_LINK} target="_blank" href={LICENSE_AGREEMENT_LINK}>
                {i18nRaw('Лицензионного соглашения (п.5.10)')}
              </Link>
            ),
            privacyPolicy: (
              <Link key={PRIVACY_POLICY_LINK} target="_blank" href={PRIVACY_POLICY_LINK}>
                {i18nRaw('Пользовательского соглашения (п.2.8.4)')}
              </Link>
            ),
          },
        )}
      </Text>
    </>
  );
};
