import { ReactNode, VFC } from 'react';

import { BillsDocumentType, GibddDataQuery, useGibddDataQuery } from '@client/shared/api/graphql';
import { Car, DriverLicenseIcon, EditOutline, Plus } from '@client/shared/icons';
import { List, ListItemLink, Section, Skeleton, Text } from '@client/shared/ui-kit';

import { useGibddPaths } from '../../libs';
import { GibddHeader } from '../GibddHeader';
import { i18n } from './i18n';

const ICONS: Record<string, ReactNode> = {
  DRIVER_LICENSE: <DriverLicenseIcon />,
  VEHICLE_REGISTRATION_CERTIFICATE: <Car />,
};

type Document = GibddDataQuery['billsDocuments'][0];

const DocumentItem = ({ document_id, title, value, type }: Document): ReactNode => {
  const { gibddDocument } = useGibddPaths();

  return (
    <ListItemLink
      href={gibddDocument(document_id)}
      plateColor="blue-gradient100"
      key={document_id}
      icon={ICONS[type]}
      label={title}
      text={value}
      after={<EditOutline />}
    />
  );
};

export const GibddSettings: VFC = () => {
  const { loading, data, error } = useGibddDataQuery();
  const { gibddAddVrc, gibddAddDl, gibdd } = useGibddPaths();

  if (loading) {
    return (
      <Skeleton height="240" width="100%">
        <rect width="100%" height="60" rx="24" />
        <rect y="80" width="100%" height="60" rx="24" />
        <rect y="160" width="100%" height="60" rx="24" />
      </Skeleton>
    );
  }

  if (!data || error) {
    return null;
  }

  return (
    <Section>
      <GibddHeader backPath={gibdd} />
      <Text variant="heading-s">{i18n('Машины')}</Text>
      <List>
        {data.billsDocuments
          .filter(({ type }) => type === BillsDocumentType.VehicleRegistrationCertificate)
          .map(DocumentItem)}
        <ListItemLink icon={<Plus />} text={i18n('Добавить СТС')} href={gibddAddVrc} />
      </List>

      <Text variant="heading-s">{i18n('Водители')}</Text>
      <List>
        {data.billsDocuments
          .filter(({ type }) => type === BillsDocumentType.DriverLicense)
          .map(DocumentItem)}
        <ListItemLink icon={<Plus />} text={i18n('Добавить ВУ')} href={gibddAddDl} />
      </List>
    </Section>
  );
};
