import cx from 'classnames';
import { useRouter } from 'next/router';
import { VFC, useEffect, useMemo, useRef } from 'react';

import { GibddBill, getBillsSum, getFee } from '@client/features/gibdd';
import { useYandexPay } from '@client/features/yandex-pay/useYandexPay';
import { Spinner, Text } from '@client/shared/ui-kit';
import { formatPrice } from '@shared/helpers';

import { i18n } from './i18n';

import styles from './YandexPayButton.module.css';

interface YandexPayButtonProps {
  bills: GibddBill[];
  sticky?: boolean;
}

const MAX_SELECTED = 100;

export const YandexPayButton: VFC<YandexPayButtonProps> = ({ bills, sticky }) => {
  const router = useRouter();

  const { isYandexPayReady, updateYandexPay, removeYandexPay } = useYandexPay();
  const buttonRef = useRef<HTMLDivElement>(null);

  const hasBills = useMemo(() => bills.length > 0, [bills]);
  const billsOverflow = useMemo(() => bills.length > MAX_SELECTED, [bills]);
  const buttonReady = isYandexPayReady && hasBills && !billsOverflow;

  const amount = useMemo<number>(() => getBillsSum(bills), [bills]);
  const fee = useMemo<number>(() => getFee(bills), [bills]);
  const amountWithFee = useMemo<number>(
    () => (hasBills ? amount + fee : amount),
    [amount, fee, hasBills],
  );

  useEffect(() => {
    if (buttonRef.current) {
      // Пришли с редиректа Pay, а значит транзакция в процессе.
      if (window.localStorage.getItem('order-id') && router.query.__YP__) {
        updateYandexPay(buttonRef.current, []);
      } else if (!billsOverflow && bills) {
        updateYandexPay(buttonRef.current, bills);
      }
    }
  }, [updateYandexPay, bills, buttonRef, removeYandexPay, billsOverflow, router.query.__YP__]);

  useEffect(() => removeYandexPay, [removeYandexPay]);

  return (
    <div className={cx(styles.root, { [styles.sticky]: sticky })}>
      {buttonReady && (
        <div className={styles.info}>
          <Text variant="heading-s">
            {i18n('Итого:')} {formatPrice(amountWithFee, 'RUB', false, { signed: false })}
          </Text>
          <Text className={styles.divider} variant="text-s" color="secondary">
            {' • '}
          </Text>
          <Text variant="text-s" color="secondary">
            {i18n('включая комиссию')} {formatPrice(fee, 'RUB', false, { signed: false })}
          </Text>
        </div>
      )}
      {billsOverflow && (
        <Text variant="heading-s">
          {i18n('Выберите не больше {count} штрафов', { count: MAX_SELECTED })}
        </Text>
      )}
      {!hasBills && <Text variant="heading-s">{i18n('Выберите счета для оплаты')}</Text>}
      {!buttonReady && (
        <div className={styles.spiner}>
          <Spinner size={32} />
        </div>
      )}
      <div ref={buttonRef} className={cx(styles.button, { [styles.buttonVisible]: buttonReady })} />
    </div>
  );
};
