import { VFC, useCallback } from 'react';

import {
  AuthType,
  useDisableLoginWithQrMutation,
  useEnableLoginWithQrMutation,
  useLoginWithQrQuery,
} from '@client/shared/api/graphql';
import checkFilledSrc from '@client/shared/assets/check-filled.png';
import crossFilledSrc from '@client/shared/assets/cross-filled.png';
import { Qr } from '@client/shared/icons';
import { Hint, ImageIcon, List, ListItemToggle, Section } from '@client/shared/ui-kit';

import { i18n } from './i18n';

export const LoginWithQR: VFC = () => {
  const { loading, data, error } = useLoginWithQrQuery();
  const [enableLoginWithQR, { loading: qrEnabling }] = useEnableLoginWithQrMutation();
  const [disableLoginWithQR, { loading: qrDisabling }] = useDisableLoginWithQrMutation();
  const isQRDisabled = qrEnabling || qrDisabling;

  const handleChangeLoginWithQR = useCallback(() => {
    const canLoginWithQR = data?.account.canLoginWithQR;

    if (typeof canLoginWithQR !== 'boolean') {
      return;
    }

    canLoginWithQR ? disableLoginWithQR() : enableLoginWithQR();
  }, [data?.account.canLoginWithQR, enableLoginWithQR, disableLoginWithQR]);

  if (error) {
    return null;
  }

  const { authType = null, canLoginWithQR = false } = data?.account || {};
  const isNotSetAuthType = authType === null;
  const isSmsOnly = authType === AuthType.SmsOnly;
  const isYandexKey = authType === AuthType.YandexKey;
  const showQRControl = !isNotSetAuthType && !isSmsOnly && !isYandexKey;

  return (
    <Section>
      <Section.Title>{i18n('Вход с помощью QR-кода')}</Section.Title>
      <Section.Description>
        {i18n('Отсканируйте QR-код телефоном, чтобы войти в аккаунт в один клик')}
      </Section.Description>

      {isYandexKey && (
        <Hint
          before={<ImageIcon src={checkFilledSrc} size={32} />}
          text={i18n('При входе с Ключом QR-код всегда доступен')}
          variant="info"
        />
      )}

      {!loading && (isNotSetAuthType || isSmsOnly) && (
        <Hint
          before={<ImageIcon src={crossFilledSrc} size={32} />}
          // TODO: Уточнить перевод
          text={i18n('При текущем способе входа нельзя войти с QR-кодом')}
          variant="negative"
        />
      )}

      {showQRControl && (
        <List>
          <ListItemToggle
            icon={<Qr />}
            text={i18n('Входить с QR')}
            isChecked={canLoginWithQR}
            disabled={isQRDisabled}
            onPress={handleChangeLoginWithQR}
          />
        </List>
      )}
    </Section>
  );
};
