import { useRouter } from 'next/router';
import { ChangeEventHandler, VFC, useCallback, useState } from 'react';

import { useEdadealData, usePaymentsContext } from '@client/features/finance/api/store';
import { Payment } from '@client/shared/api/graphql';
import { List as ListIcon, Receipts } from '@client/shared/icons';
import { logProps } from '@client/shared/libs/metrika';
import { sendParams } from '@client/shared/libs/metrika';
import { useIdPaths } from '@client/shared/paths';
import { List, ListItemLink, RadioButton, Section } from '@client/shared/ui-kit';

import { EdadealLanding } from './components/EdadealLanding';
import { Filters } from './components/Filters';
import { PaymentDetails } from './components/PaymentDetails';
import { PaymentsList } from './components/PaymentsList';
import { PaymentHistoryProvider } from './contexts/paymentDetails';
import { i18n } from './i18n';

import styles from './PaymentsHistory.module.css';

const URLS: Record<string, string> = {
  yandex: '/pay/history',
  receipts: '/pay/history/receipts',
};

const getUrlBySlug = (value: string): string => {
  return URLS[value] || URLS.yandex;
};

interface IPaymentsHistoryProps {
  onlyYandex?: boolean;
  shortHistory?: boolean;
}

export const PaymentsHistory: VFC<IPaymentsHistoryProps> = ({
  onlyYandex,
  shortHistory = false,
}) => {
  const idPaths = useIdPaths();
  const [historyCount, setHistoryCount] = useState<number>(0);
  const [details, setDetails] = useState<Payment>();
  const router = useRouter();
  const { query } = router;
  const edadealData = useEdadealData();
  const { data } = usePaymentsContext();
  const isFamilyMode = query.family === '1';
  const userFilter =
    isFamilyMode && !query.user ? data?.familyData?.map(({ uid }) => uid).join(',') : query.user;

  const value = router.pathname === '/pay/history' || onlyYandex ? 'yandex' : 'receipts';

  const onChangeTab: ChangeEventHandler<HTMLInputElement> = useCallback(
    (e) => {
      sendParams('click.finance.tabs', { value: e.target.value });
      router.push(getUrlBySlug(e.target.value), undefined, { scroll: false });
    },
    [router],
  );

  const isEdadealTab = value === 'receipts';

  let content = <></>;

  if (shortHistory) {
    content = (
      <>
        <PaymentsList />
        <List>
          {Boolean(historyCount) && (
            <ListItemLink
              icon={<ListIcon />}
              text={i18n('Все платежи')}
              href={idPaths.payHistory}
              {...logProps('payments-short.history.link')}
            />
          )}
          <ListItemLink
            icon={<Receipts />}
            text={i18n('Мои чеки')}
            href={idPaths.payReceipts}
            {...logProps('payments-short.history.receipts.link')}
          />
        </List>
      </>
    );
  } else if (isEdadealTab) {
    content = edadealData.data?.hasBinding ? <PaymentsList alias="receipts" /> : <EdadealLanding />;
  } else {
    content = (
      <>
        <Filters />
        <PaymentsList alias={query.alias as string} user={userFilter as string} />
      </>
    );
  }

  const openPopup = useCallback((data: any) => {
    setDetails(data);
  }, []);

  const closePopup = useCallback(() => {
    setDetails(undefined);
  }, []);

  return (
    <PaymentHistoryProvider
      value={{
        paymentDetails: { data: details, openPopup, closePopup },
        paymentList: { setCount: setHistoryCount },
        isShort: shortHistory,
      }}
    >
      <Section>
        <Section.Title anchor="payments-history">{i18n('История платежей')}</Section.Title>
        {!shortHistory && !onlyYandex && (
          <RadioButton
            className={styles.tabs}
            size="l"
            variant="outline"
            value={value}
            options={[
              { value: 'yandex', children: i18n('Яндекс') },
              { value: 'receipts', children: i18n('Мои чеки') },
            ]}
            onChange={onChangeTab}
          />
        )}
        {content}
      </Section>
      <PaymentDetails />
    </PaymentHistoryProvider>
  );
};
