import cx from 'classnames';
import { useRouter } from 'next/router';
import React, { ElementType } from 'react';

import { useEdadealData } from '@client/features/finance/api/store';
import { useEdadealConnectUrl } from '@client/features/finance/hooks';
import { useEnvContext } from '@client/shared/libs/env';
import { logProps } from '@client/shared/libs/metrika';
import { Button, Section, SectionStack } from '@client/shared/ui-kit';

import { Divider } from './Divider/Divider';
import { Spoiler } from './Spoiler';
import { i18n, i18nRaw } from './i18n';

import styles from './EdadealLanding.module.css';

const PARTNERS = ['ashan', 'mcdonalds', 'diksy', 'market', 'perekrestok', 'ozon', 'lenta'];

interface ILinkProps {
  className?: string;
  href?: string;
}

const Link: React.FC<ILinkProps> = ({ children, href }) => {
  return (
    <a className={styles.link} href={href} target="_blank" rel="noreferrer">
      {children}
    </a>
  );
};

export const EdadealLanding = () => {
  const { data, isLoading } = useEdadealData();
  const {
    ua: { isTouch },
  } = useEnvContext();
  const router = useRouter();
  const connectUrl = useEdadealConnectUrl({ url: data?.url });

  if (isLoading) {
    return null;
  }

  function onPressConnect() {
    router.push({ hash: 'receipts-popup-promo' });
  }

  const buttonProps = isTouch
    ? {
        as: 'a' as ElementType,
        href: connectUrl,
        target: '_blank',
      }
    : { onPress: onPressConnect };

  return (
    <SectionStack className={styles.root}>
      <Section className={styles.block}>
        <h2 className={styles.headerTitle}>
          {i18n('Получайте кешбэк за{nbsp}покупки', { nbsp: '\u00A0' })}
        </h2>
        <p className={styles.description}>
          {i18n('Все чеки будут сохраняться в истории платежей, а{nbsp}мы начислим кешбэк', {
            nbsp: '\u00A0',
          })}
        </p>
        <Button
          className={styles.button}
          size="xl"
          variant="action"
          {...buttonProps}
          {...logProps('edadeal-landing.button')}
        >
          {i18n('Хочу кешбэк')}
        </Button>
      </Section>
      <Section className={styles.block}>
        <h3 className={styles.title}>{i18n('Зачем сохранять чеки?')}</h3>
        <div className={styles.promo}>
          <div className={cx(styles.image, styles.image_type_receipt)} />
          <p className={styles.text}>
            {i18nRaw('Если в чеке будут {link}, вы получите вознаграждение.', {
              link: (
                <Link key="edadeal-link" href="https://edadeal.ru/cashbacks">
                  {i18n('товары из списка кешбэк-акций')}
                </Link>
              ),
            })}
          </p>
        </div>
        <Divider />
        <div className={cx(styles.promo, styles.promo_align_center)}>
          <div className={cx(styles.image, styles.image_type_bag)} />
          <p className={styles.text}>
            {i18n(
              'Вы всегда сможете найти где, что{nbsp}и{nbsp}когда покупали в{nbsp}истории платежей.',
              { nbsp: '\u00A0' },
            )}
          </p>
        </div>
        <div className={styles.partners}>
          {PARTNERS.map((item) => (
            <span key={item} className={cx(styles.partner, styles['partner_' + item])} />
          ))}
        </div>
        <Divider />
        <div className={styles.promo}>
          <div className={cx(styles.image, styles.image_type_cart)} />
          <p className={styles.text}>
            {i18n(
              'Пригодится при обмене или возврате: электронный чек является подтверждением вашей покупки, как и бумажный.',
            )}
          </p>
        </div>
      </Section>
      <Section className={styles.block}>
        <h3 className={styles.title}>{i18n('Остались вопросы?')}</h3>
        <Spoiler
          className={styles.spoiler}
          summary={i18n('Как мои чеки попадут в историю платежей?')}
        >
          {i18n(
            'При регистрации укажите номер телефона и почту, которые используете для покупок и регистрации карт лояльности. Сделайте покупку. При оформлении заказа укажите те же данные или используйте карту лояльности. В течение 24 часов чек с вашей покупкой появится в Яндекс ID.',
          )}
        </Spoiler>
        <Spoiler className={styles.spoiler} summary={i18n('Я получу доступ ко всем своим чекам?')}>
          {i18n(
            'Будут доступны только те чеки, которые были получены после подключения к сервису «Мои чеки онлайн». Более ранние чеки не попадут в историю. Электронные чеки выдают более 10 000 организаций в России: сервисы Яндекса, популярные интернет-магазины, мобильные операторы, торговые сети, интернет-провайдеры, доставщики еды, сервисы по прокату и аренде, и многие другие.',
          )}
        </Spoiler>
        <Spoiler
          className={styles.spoiler}
          summary={i18n('Можно ли получать чеки из оффлайн магазинов?')}
        >
          {i18nRaw(
            'На данный момент электронные чеки выдают в Пятерочке, Перекрестке, ВкусВилле, аптеках Неофарм и в некоторых других крупных торговых сетях. Для получения чеков от этих продавцов необходимо включить специальную опцию в личном кабинете в приложении торговой сети и использовать карту лояльности при покупке. Оффлайн магазинов, где есть возможность отправить чек в электронном виде, постепенно становится всё больше. Вы можете посмотреть актуальный список по {link}.',
            {
              link: (
                <Link key="markets-link" href="https://lkdr.nalog.ru/partners/markets">
                  {i18n('ссылке')}
                </Link>
              ),
            },
          )}
        </Spoiler>
        <Spoiler className={styles.spoiler} summary={i18n('Как отключить передачу чеков?')}>
          {i18nRaw(
            'Вы можете в любой момент отключить доступ Едадил к вашим чекам. Перейдите по {link} и кликните по кнопке "Отключить" для партнера Едадил.',
            {
              link: (
                <Link key="partners-link" href="https://lkdr.nalog.ru/partners">
                  {i18n('ссылке')}
                </Link>
              ),
            },
          )}
        </Spoiler>
        <Spoiler className={styles.spoiler} summary={i18n('Где увидеть все кешбэк-акции?')}>
          {i18nRaw('В разделе {link} в сервисе Едадил.', {
            link: (
              <Link key="cashbacks-link" href="https://edadeal.ru/cashbacks">
                {i18n('«Кешбэк»')}
              </Link>
            ),
          })}
        </Spoiler>
        <Spoiler className={styles.spoiler} summary={i18n('Сколько времени занимает подключение?')}>
          {i18n(
            'Обычно мы обрабатываем подключение за 20 минут. Иногда, из-за технических задержек, время подключения может увеличиться.',
          )}
        </Spoiler>
        <Spoiler className={styles.spoiler} summary={i18n('Как быстро начислится кешбэк?')}>
          {i18n(
            'Чеки обрабатываются за 1–3 дня. Если одновременно поступило много чеков — сроки проверки могут увеличиться.',
          )}
        </Spoiler>
      </Section>
    </SectionStack>
  );
};
