import { useRouter } from 'next/router';
import { VFC, useCallback } from 'react';

import { usePaymentsContext } from '@client/features/finance/api/store';
import { useEnvContext } from '@client/shared/libs/env';
import { sendParams } from '@client/shared/libs/metrika';
import { getAvatarUrl, useAvatarsPath } from '@client/shared/paths';
import { ServiceKeyType, i18nServices } from '@client/shared/translations/services';

import { FilterButton } from '../FilterButton';
import { i18n } from './i18n';

import styles from './Filters.module.css';

export const Filters: VFC = () => {
  const {
    viewer: { avatarId, uid: userId },
  } = useEnvContext();
  const { data, isLoading } = usePaymentsContext();
  const router = useRouter();
  const selectedAlias = router.query.alias as string;
  const selectedUser = router.query.user as string;
  const isFamilyMode = router.query.family === '1';
  const avatarHost = useAvatarsPath();

  const onClickFilter = useCallback(
    ({ alias, user }) =>
      () => {
        const query: Record<string, string | undefined> = {
          ...router.query,
          alias: alias || selectedAlias,
          user: user || selectedUser,
        };

        if (alias === selectedAlias) {
          delete query.alias;
        }

        if (user === selectedUser) {
          delete query.user;
        }

        if (alias === 'plus') {
          delete query.user;
        }

        if (user && selectedAlias === 'plus') {
          delete query.alias;
        }

        // Не шлем в метрику uid-ы, интересен только факт нажатия на семейный фильтр
        sendParams('click.finance.filter', { alias, user: user ? true : undefined });

        router.push({ query }, undefined, {
          scroll: false,
        });
      },
    [router, selectedAlias, selectedUser],
  );

  if (isLoading || !data.services || data.services.length < 2) {
    return null;
  }

  const hasFamilyFilters = data.familyData && data.familyData.length > 0;

  return (
    <div className={styles.root}>
      {hasFamilyFilters && !isFamilyMode && (
        <FilterButton
          text={i18n('Я')}
          iconUrl={getAvatarUrl(avatarHost, avatarId)}
          onPress={onClickFilter({ user: userId })}
          checked={selectedUser === userId}
        />
      )}
      {data.familyData?.map(({ uid, name, avatar }) => (
        <FilterButton
          key={uid}
          text={name}
          iconUrl={getAvatarUrl(avatarHost, avatar)}
          onPress={onClickFilter({ user: uid })}
          checked={selectedUser === uid}
        />
      ))}
      {hasFamilyFilters && <div className={styles.divider}>•</div>}
      {data.services.map(({ id, name, iconUrl, i18nKeyName }) => {
        const text = (i18nKeyName && i18nServices(i18nKeyName as ServiceKeyType)) || name;

        return (
          <FilterButton
            key={id}
            text={text}
            iconUrl={iconUrl}
            onPress={onClickFilter({ alias: id })}
            checked={selectedAlias === id}
          />
        );
      })}
    </div>
  );
};
