import { VFC } from 'react';

import { usePaymentsContext } from '@client/features/finance/api/store';
import { getAvatarUrl, useAvatarsPath } from '@client/shared/paths';
import { i18nServices } from '@client/shared/translations/services';
import { Dialog } from '@client/shared/ui-kit';
import { formatPlus, formatPrice } from '@shared/helpers';

import { usePaymentHistoryContext } from '../../contexts/paymentDetails';
import { getDescription } from '../../helpers/description';
import * as helpers from '../../helpers/order';
import { getServiceName } from '../../helpers/serviceName';
import { PaymentHeader } from './components/PaymentHeader';
import { PaymentInfo } from './components/PaymentInfo/PaymentInfo';

import styles from './PaymentDetails.module.css';

export const PaymentDetails: VFC = () => {
  const {
    paymentDetails: { data, closePopup },
  } = usePaymentHistoryContext();
  const context = usePaymentsContext();
  const avatarHost = useAvatarsPath();

  if (!data || context.isLoading) {
    return null;
  }

  const isPlus = helpers.isPlusTopup(data);
  const isPlusTransaction = helpers.isPlusTransaction(data);
  const isRefund = data.status === 'refunded';
  const initiator = data.initiatorUid
    ? context.data.familyData?.find(({ uid }) => data.initiatorUid === uid)
    : undefined;

  let iconUrl = isPlus ? context.data.plusServiceData?.iconUrl : data?.service.iconUrl;
  let serviceName = isPlus ? i18nServices('plus_points') : getServiceName(data);
  let subIcon;

  if (initiator?.avatar) {
    subIcon = iconUrl;

    iconUrl = getAvatarUrl(avatarHost, initiator.avatar);
  }

  if (initiator?.name) {
    serviceName = [initiator.name, serviceName].join(' • ');
  }

  if (data.source === 'yandexpay') {
    serviceName = i18nServices('yandexpay_purchase');
  }

  return (
    <Dialog hasClose={false} visible={Boolean(data)} onClose={closePopup} className={styles.root}>
      <div className={styles.content}>
        <PaymentHeader
          status={data?.status}
          serviceName={serviceName}
          iconUrl={iconUrl}
          subIconUrl={subIcon}
          backgroundImage={isPlus ? undefined : data?.service.splashUrl}
          description={isPlus ? getServiceName(data) : getDescription(data)}
          isPlus={isPlus}
          onClose={closePopup}
          total={
            isPlusTransaction && data.plus
              ? formatPlus(data.plus, isRefund, { leadPlus: true })
              : formatPrice(data.total, data?.currency, isRefund)
          }
        />
        <PaymentInfo
          purchaseToken={data.purchaseToken}
          created={data.created}
          items={data.items}
          paymentTypes={data.paymentTypes}
          isRefund={isRefund}
          currency={data.currency}
          serviceUrl={data.service.url}
          supportUrl={data.service.help}
          receiptsEnabled={!isPlus && data.receiptsEnabled}
          receiptUrl={data.isReceipt ? data.receiptUrl : undefined}
          serviceKey={data.service.i18nKeyName}
        />
      </div>
    </Dialog>
  );
};
