import { VFC, useCallback, useState } from 'react';

import { usePaymentReceiptsQuery } from '@client/shared/api/graphql';
import { ArrowLongForward, ChatBubble } from '@client/shared/icons';
import { logProps } from '@client/shared/libs/metrika';
import { Button } from '@client/shared/ui-kit';
import {
  getDetailByAlias,
  isHiddenAlias,
} from '@client/widgets/PaymentsHistory/helpers/description';
import { Order } from '@client/widgets/PaymentsHistory/types';
import { formatPrice } from '@shared/helpers';
import { useDateFormatter } from '@use-platform/react';

import { PaymentType } from './components/PaymentType';
import { ReceiptItem, ReceiptSkeleton, ReceiptsList } from './components/ReceiptsList';
import { i18n } from './i18n';

import styles from './PaymentInfo.module.css';

interface PaymentInfoProps {
  purchaseToken: string;
  className?: string;
  created: Order['created'];
  items: Order['items'];
  paymentTypes: Order['paymentTypes'];
  currency: string;
  isRefund: boolean;
  supportUrl?: string | null;
  serviceUrl?: string | null;
  receiptsEnabled?: boolean;
  receiptUrl?: string | null;
  serviceKey?: string | null;
}

const SQUEEZE_LENGTH = 5;

export const PaymentInfo: VFC<PaymentInfoProps> = ({
  created,
  items,
  paymentTypes,
  isRefund,
  currency,
  serviceUrl,
  supportUrl,
  purchaseToken,
  receiptsEnabled,
  receiptUrl,
  serviceKey,
}) => {
  const [expandItems, setExpandItems] = useState(false);
  const dateFormatter = useDateFormatter({ timeStyle: 'short', dateStyle: 'long' });
  const receipts = usePaymentReceiptsQuery({
    variables: { purchaseToken },
    skip: !receiptsEnabled || Boolean(receiptUrl),
  });

  let displayedItems = items?.filter(({ hidden }) => !hidden) || [];

  // Не хотим показывать Доставку, Сервисный сбор и т.п. в деталях, если кроме них ничего нет
  if (displayedItems.length === 1 && isHiddenAlias(displayedItems[0].alias)) {
    displayedItems = [];
  }

  const needToSqueeze = displayedItems.length > SQUEEZE_LENGTH + 1 && !expandItems;

  const onExpand = useCallback(() => {
    setExpandItems(true);
  }, []);

  return (
    <div className={styles.root}>
      <section className={styles.section}>
        <div className={styles.title}>{dateFormatter.format(new Date(created))}</div>

        {Array.from(paymentTypes || [])
          .sort((a, b) => b.price - a.price)
          .map(({ method, currency, price, account }) => (
            <PaymentType
              className={styles.payment}
              key={method}
              method={method}
              currency={currency}
              account={account}
              price={price}
              isRefund={isRefund}
            />
          ))}
      </section>

      {displayedItems?.length > 0 && (
        <section className={styles.section}>
          <div className={styles.title}>{i18n('Детали платежа')}</div>

          {displayedItems
            .sort((a, b) => b.price - a.price)
            .slice(0, needToSqueeze ? SQUEEZE_LENGTH : undefined)
            .map(({ name, price, alias }) => (
              <div key={name + price} className={styles.item}>
                <div className={styles.itemName}>{getDetailByAlias(alias, created) || name}</div>
                <div className={styles.itemPrice}>
                  {formatPrice(price, currency, isRefund, { roundIfNeeded: false, signed: false })}
                </div>
              </div>
            ))}

          {needToSqueeze && (
            <button className={styles.expand} onClick={onExpand}>
              {i18n('Показать еще {count}', { count: displayedItems.length - SQUEEZE_LENGTH })}
            </button>
          )}
        </section>
      )}

      {receipts.loading && (
        <section className={styles.section}>
          <ReceiptSkeleton />
        </section>
      )}

      {!receipts.loading &&
        receipts.data?.receiptsByToken &&
        receipts.data.receiptsByToken.length > 0 && (
          <section className={styles.section}>
            <div className={styles.title}>{i18n('Чеки')}</div>
            <ReceiptsList
              receipts={receipts.data.receiptsByToken}
              currency={currency}
              service={serviceKey}
            />
          </section>
        )}

      {receiptUrl && (
        <section className={styles.section}>
          <div className={styles.title}>{i18n('Чеки')}</div>
          <ReceiptItem
            url={receiptUrl}
            label={i18n('Открыть чек')}
            {...logProps('finance.payment-details.edadeal-receipt')}
          />
        </section>
      )}

      {(serviceUrl || supportUrl) && (
        <div className={styles.buttons}>
          {serviceUrl && (
            <div className={styles.extendedButton}>
              <Button
                size="xl"
                className={styles.button}
                href={serviceUrl}
                target="_blank"
                aria-label={i18n('На сервис')}
                as="a"
                {...logProps('finance.payment-details.to-service')}
              >
                <ArrowLongForward />
              </Button>
              <div className={styles.buttonText} aria-hidden="true">
                {i18n('На сервис')}
              </div>
            </div>
          )}
          {supportUrl && (
            <div className={styles.extendedButton}>
              <Button
                size="xl"
                className={styles.button}
                href={supportUrl}
                target="_blank"
                as="a"
                aria-label={i18n('Поддержка')}
                {...logProps('finance.payment-details.support')}
              >
                <ChatBubble />
              </Button>
              <div className={styles.buttonText} aria-hidden="true">
                {i18n('Поддержка')}
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
};
