import cx from 'classnames';
import { ReactNode, VFC } from 'react';

import {
  AmericanexrpessDefaultIcon,
  ApplepayoutlinedDefaultIcon,
  ChinaunionpayDefaultIcon,
  CreditCardOutline,
  Edadeal,
  GooglepayoutlinedDefaultIcon,
  MastercardDefaultIcon,
  MirDefaultIcon,
  PlusIcon,
  Sbp,
  VisaDefaultIcon,
} from '@client/shared/icons';
import { i18nServices } from '@client/shared/translations/services';
import { PlusValue } from '@client/widgets/PaymentsHistory/components/PlusValue';
import {
  PaymentSystem,
  getPaymentMethodKey,
  getPaymentSystemByMask,
} from '@client/widgets/PaymentsHistory/helpers/order';
import { formatPlus, formatPrice } from '@shared/helpers';

import { i18n } from './i18n';

import styles from './PaymentType.module.css';

interface PaymentTypeProps {
  method: string;
  price: number;
  currency?: string | null;
  account?: string | null;
  className?: string;
  isRefund: boolean;
}

const ICONS_METHOD: Record<string, ReactNode> = {
  yandex_account_withdraw: <PlusIcon className={styles.icon} width={24} height={24} />,
  yandex_account_topup: <PlusIcon className={styles.icon} width={24} height={24} />,
  card: <CreditCardOutline className={styles.icon} />,
  cashback_edadeal: <Edadeal className={styles.icon} width={24} height={24} />,
};

const ICONS_SYSTEM: Record<PaymentSystem, ReactNode> = {
  MasterCard: <MastercardDefaultIcon />,
  Visa: <VisaDefaultIcon />,
  Mir: <MirDefaultIcon />,
  AmericanExpress: <AmericanexrpessDefaultIcon />,
  UnionPay: <ChinaunionpayDefaultIcon />,
  UnknownCard: null,
};

const ADDITIONAL_ICONS: Record<string, ReactNode> = {
  google_pay: <GooglepayoutlinedDefaultIcon />,
  apple_pay: <ApplepayoutlinedDefaultIcon />,
  sbp_qr: <Sbp height={22} />,
};

const PAYMENT_NAMES: Record<string, () => string> = {
  Mir: () => i18n('МИР'),
  AmericanExpress: () => 'American Express',
  Visa: () => 'VISA',
  MasterCard: () => 'Master Card',
  UnionPay: () => 'Union Pay',
  paying: () => i18n('Успешная оплата'),
  card: () => i18n('Оплата картой'),
  yamoney_wallet: () => i18n('Оплата ЮMoney'),
  cash: () => i18n('Оплата наличными'),
  new_promocode: () => i18n('Оплата промокодом'),
  spasibo: () => i18n('Оплата бонусами от СберБанка'),
  yandex_account_withdraw: () => i18nServices('plus_points'),
  yandex_account_topup: () => i18nServices('plus_points'),
  afisha_certificate: () => i18n('Оплата промокодом Афишы'),
  google_pay: () => 'Google Pay',
  apple_pay: () => 'Apple Pay',
  virtual_kinopoisk: () => i18n('Скидка'),
  sbp_qr: () => i18n('Оплата СБП'),
  cashback_edadeal: () => i18n('Кешбэк Едадила'),
};

const getPaymentName = (key: string) => {
  return PAYMENT_NAMES[getPaymentMethodKey(key)];
};

export const PaymentType: VFC<PaymentTypeProps> = ({
  method,
  price,
  className,
  account,
  isRefund,
  currency,
}) => {
  const isPaymentByCard = method === 'card';
  const isPlusPoints = method === 'yandex_account_topup' || method === 'yandex_account_withdraw';
  const isEdadealCashback = method === 'cashback_edadeal';

  let label = '';
  let additional: ReactNode = null;

  if (isPaymentByCard) {
    const paymentSystem = account ? getPaymentSystemByMask(account) : undefined;
    const systemName =
      paymentSystem && paymentSystem !== 'UnknownCard'
        ? PAYMENT_NAMES[paymentSystem]()
        : PAYMENT_NAMES.card();

    label = account ? `${systemName} •••• ${account.slice(-4)}` : PAYMENT_NAMES.paying();
    additional = paymentSystem ? ICONS_SYSTEM[paymentSystem] : null;
  }

  if (isPlusPoints) {
    label = getPaymentName(method)();
    const value = formatPlus(method === 'yandex_account_withdraw' ? -price : price, isRefund, {
      leadPlus: true,
    });

    additional = <PlusValue>{value}</PlusValue>;
  }

  if (!label) {
    label = getPaymentName(method)();
    additional =
      ADDITIONAL_ICONS[method] || formatPrice(price, currency, isEdadealCashback, { signed: true });
  }

  return (
    <div className={cx(styles.root, className)}>
      {ICONS_METHOD[method] || ICONS_METHOD.card}
      <div className={styles.label}>{label}</div>
      <div className={cx(styles.additional, { [styles.edadeal]: isEdadealCashback })}>
        {additional}
      </div>
    </div>
  );
};
