import cx from 'classnames';
import React, { VFC, useCallback, useState } from 'react';

import { Receipt } from '@client/shared/api/graphql';
import { ArrowShortForward, Receipt as ReceiptIcon, Receipts } from '@client/shared/icons';
import { logProps } from '@client/shared/libs/metrika';
import { Skeleton } from '@client/shared/ui-kit';
import { formatPrice } from '@shared/helpers';
import { useDateFormatter } from '@use-platform/react';

import { i18n } from './i18n';

import styles from './ReceiptsList.module.css';

interface ReceiptsListProps {
  receipts: Receipt[];
  currency: string;
  service?: string | null;
}

interface ReceiptItemProps extends Receipt {
  onClick?(): void;
  currency?: string;
  label?: string;
}

export const ReceiptSkeleton = () => {
  return (
    <Skeleton>
      <rect width="24" height="24" rx="12" />
      <rect x="36" width="180" height="24" rx="12" />
    </Skeleton>
  );
};

export const ReceiptItem: VFC<ReceiptItemProps> = ({
  timestamp,
  amount,
  currency,
  url,
  label,
  type,
  ...otherProps
}) => {
  const formatDate = useDateFormatter({ dateStyle: 'short', timeStyle: 'short' });

  return (
    <a className={styles.item} href={url} target="_blank" rel="noreferrer" {...otherProps}>
      <ReceiptIcon className={styles.itemIcon} />
      <div className={styles.itemText}>
        {label}
        {amount?.map((item) => formatPrice(item, currency, false, { signed: false })).join(' + ')}
        {timestamp && (
          <span className={styles.time}> – {formatDate.format(new Date(timestamp))}</span>
        )}
      </div>
      <ArrowShortForward className={styles.arrow} />
    </a>
  );
};

export const ReceiptsList: VFC<ReceiptsListProps> = ({ receipts, currency, service }) => {
  const [opened, setOpened] = useState(false);

  const onExtendClick = useCallback(() => {
    setOpened(true);
  }, []);

  const [initiallyShowedReceipts, otherReceipts] = receipts.reduce<[Receipt[], Receipt[]]>(
    (acc, item) => {
      acc[item.type === 'prepayment' ? 1 : 0].push(item);

      return acc;
    },
    [[], []],
  );

  return (
    <div className={styles.root}>
      {initiallyShowedReceipts.map((item) => (
        <ReceiptItem
          key={item.timestamp}
          currency={currency}
          {...item}
          {...logProps('finance.payment-details.receipt', { service, variant: item.type })}
        />
      ))}
      {otherReceipts.length > 0 && !opened ? (
        <div className={cx(styles.item, styles.extend)}>
          <Receipts className={styles.itemIcon} />
          <div
            className={styles.itemText}
            onClick={onExtendClick}
            {...logProps('finance.payment-details.expand-receipt')}
          >
            {i18n('Показать все документы')}
          </div>
        </div>
      ) : (
        otherReceipts.map((item) => (
          <ReceiptItem
            key={item.timestamp}
            currency={currency}
            {...item}
            {...logProps('finance.payment-details.receipt', { service, variant: item.type })}
          />
        ))
      )}
    </div>
  );
};
