import { isToday, isYesterday } from 'date-fns';
import { VFC, useCallback, useEffect, useMemo, useState } from 'react';

import { usePaymentsContext } from '@client/features/finance/api/store';
import { FamilyUser, Service, usePaymentsHistoryQuery } from '@client/shared/api/graphql';
import { sendParams } from '@client/shared/libs/metrika';
import { useAvatarsPath } from '@client/shared/paths';
import { useDateFormatter } from '@use-platform/react';

import { usePaymentHistoryContext } from '../../contexts/paymentDetails';
import { dataToProps } from '../../helpers/mapper';
import { isPlusTopup } from '../../helpers/order';
import { Order } from '../../types';
import { Anchor } from '../Anchor';
import { Empty } from '../Empty';
import { PaymentItem, PaymentSkeleton } from '../PaymentItem';
import { i18n } from './i18n';

import styles from './PaymentsList.module.css';

interface ListProps {
  orders: Order[];
  plusServiceData?: Service;
  familyData?: FamilyUser[] | null;
}

const List: VFC<ListProps> = ({ orders }) => {
  const { data } = usePaymentsContext();
  const {
    paymentDetails: { openPopup },
    paymentList: { setCount },
    isShort,
  } = usePaymentHistoryContext();
  const formatDate = useDateFormatter({ dateStyle: 'long' });
  const avatarHost = useAvatarsPath();

  const onPressItem = useCallback(
    (data: Order) => () => {
      const service = data.isReceipt ? 'receipt' : data.service.i18nKeyName;
      const plusTopup = isPlusTopup(data) ? true : undefined;

      openPopup(data);
      sendParams('click.finance.payment', { service, plusTopup });
    },
    [openPopup],
  );

  const content = useMemo(() => {
    let lastDateStamp: string;

    const now = new Date();

    const contentData = orders?.reduce<Array<React.ReactElement | null>>((aux, node) => {
      if (!isShort) {
        const created = new Date(node.created);

        let datestamp = formatDate.format(created);

        if (created.getFullYear() === now.getFullYear()) {
          const parts = formatDate.formatToParts(created);

          datestamp = parts
            .map(({ value }) => value)
            .slice(0, 4)
            .join('');
        }

        if (isToday(created)) {
          datestamp = i18n('Сегодня');
        } else if (isYesterday(created)) {
          datestamp = i18n('Вчера');
        }

        if (datestamp && datestamp !== lastDateStamp) {
          aux.push(
            <h3 key={datestamp} className={styles.divider}>
              {datestamp}
            </h3>,
          );
        }

        lastDateStamp = datestamp;
      }

      aux.push(
        <PaymentItem
          onPress={onPressItem(node)}
          className={styles.item}
          key={node.orderId}
          {...dataToProps(node, {
            plusServiceIconUrl: data?.plusServiceData?.iconUrl,
            familyData: data?.familyData,
            avatarHost,
          })}
        />,
      );

      return aux;
    }, []);

    setCount(contentData.length);

    if (isShort) {
      return contentData.slice(0, 3);
    }

    return contentData;
  }, [formatDate, orders, onPressItem, avatarHost, data]);

  return <>{content}</>;
};

interface PaymentsListProps {
  alias?: string;
  user?: string;
}

export const PaymentsList: VFC<PaymentsListProps> = ({ alias, user }) => {
  const { isShort } = usePaymentHistoryContext();
  const { data, loading, fetchMore } = usePaymentsHistoryQuery({
    variables: { alias, user },
  });
  const [isUpdating, setUpdating] = useState(false);
  const [lastLogged, setLastLogged] = useState('');

  const orders = data?.paymentsData.edges;
  const isEmpty = !orders || orders.length === 0;

  useEffect(() => {
    const loggedKey = [alias, user].join('');

    if (!loading && (!lastLogged || lastLogged !== loggedKey)) {
      sendParams('show.finance.list', {
        empty: isEmpty,
        alias: alias || 'all',
        user: user ? true : undefined,
      });
      setLastLogged(loggedKey);
    }
  }, [loading]);

  const onFetchMore = useCallback(() => {
    setUpdating(true);
    sendParams('scroll.finance.list');

    fetchMore({
      variables: {
        alias,
        user,
        after: data?.paymentsData.pageInfo.endCursor,
      },
      updateQuery: (prev, { fetchMoreResult }) => {
        if (!fetchMoreResult) return prev;

        setUpdating(false);

        return {
          paymentsData: {
            edges: [...prev.paymentsData.edges, ...fetchMoreResult.paymentsData.edges],
            pageInfo: fetchMoreResult.paymentsData.pageInfo,
          },
        };
      },
    });
  }, [alias, data?.paymentsData.pageInfo.endCursor, fetchMore, user]);

  if (loading) {
    return (
      <div>
        <div className={styles.divider}>{i18n(`Сейчас загрузим...`)}</div>
        <PaymentSkeleton className={styles.item} />
        <PaymentSkeleton className={styles.item} />
        <PaymentSkeleton />
      </div>
    );
  }

  if (isEmpty) {
    return <Empty alias={alias} />;
  }

  return (
    <div className={styles.root}>
      <List orders={data.paymentsData.edges} />
      {!isShort && isUpdating && (
        <>
          <div className={styles.divider}>{i18n(`Сейчас загрузим...`)}</div>
          <PaymentSkeleton className={styles.item} />
        </>
      )}
      {!isShort && data?.paymentsData.pageInfo.hasNextPage && !isUpdating && (
        <Anchor onHit={onFetchMore} />
      )}
    </div>
  );
};
