import { Order } from '../types';
import { i18n } from './i18n';

type HourOfDay = 'night' | 'morning' | 'afternoon' | 'evening';

const hiddenAliases = ['delivery', 'service_fee'];

export function isHiddenAlias(alias?: string | null) {
  if (!alias) {
    return false;
  }

  return hiddenAliases.includes(alias);
}

const hourOfDay = (time: number): HourOfDay => {
  if (time >= 23 || time < 5) {
    return 'night';
  }

  if (time >= 5 && time < 12) {
    return 'morning';
  }

  if (time >= 12 && time < 18) {
    return 'afternoon';
  }

  return 'evening';
};

const detailAliases = {
  delivery: () => i18n('Доставка'),
  tip: () => i18n('Чаевые'),
  topup: () => i18n('Пополнение счёта'),
  fine: () => i18n('Штраф, возмещение убытков'),
  'toll-road': () => i18n('Проезд по платной дороге'),
  trip: (created: string) => {
    const period = hourOfDay(new Date(created).getHours());

    return {
      night: () => i18n('Поездка ночью'),
      morning: () => i18n('Поездка утром'),
      afternoon: () => i18n('Поездка днём'),
      evening: () => i18n('Поездка вечером'),
    }[period]();
  },
  ride: (created: string) => {
    const period = hourOfDay(new Date(created).getHours());

    return {
      night: () => i18n('Ночной заказ'),
      morning: () => i18n('Утренний заказ'),
      afternoon: () => i18n('Дневной заказ'),
      evening: () => i18n('Вечерний заказ'),
    }[period]();
  },
};

type DetailAliases = keyof typeof detailAliases;

export const getDetailByAlias = (alias: string | undefined | null, created: string): string => {
  const aliasExist = alias ? alias in detailAliases : false;

  return aliasExist ? detailAliases[alias as DetailAliases](created) : '';
};

const getDriveDescription = (data: Order): string => {
  const productName = data.items && data.items.length > 0 ? data.items[0].name : undefined;
  const alias = data.items && data.items.length > 0 ? data.items[0].alias : undefined;

  if (!productName) {
    return '';
  }

  return getDetailByAlias(alias, data.created) || productName;
};

const getGoDescription = (data: Order) => {
  const product = data.items ? data.items[0] : undefined;
  const alias = data.items && data.items.length > 0 ? data.items[0].alias : undefined;

  if (!product) {
    return '';
  }

  const period = hourOfDay(new Date(data.created).getHours());

  if (product.name === i18n('Услуги курьерской доставки')) {
    return {
      night: () => i18n('Доставка ночью'),
      morning: () => i18n('Доставка утром'),
      afternoon: () => i18n('Доставка днём'),
      evening: () => i18n('Доставка вечером'),
    }[period]();
  }

  return getDetailByAlias(alias, data.created);
};

const getDefaultDescription = (data: Order) => {
  if (!data.items) {
    return '';
  }

  const goods = data.items
    .slice()
    .sort((a, b) => b.price - a.price)
    .filter(({ alias }) => !isHiddenAlias(alias));

  const goodsLength = goods?.length;

  let description = goods
    ?.filter(({ hidden }) => !hidden)
    .map((item) => getDetailByAlias(item.alias, data.created) || item.name)
    .join(', ');

  if (goodsLength && goodsLength > 1) {
    const prefix =
      data.service.i18nKeyName === 'afisha'
        ? i18n('билетов', { count: goodsLength })
        : i18n('товаров', { count: goodsLength });

    description = [`${goodsLength} ${prefix}`, description].filter(Boolean).join(': ');
  }

  return description;
};

type GetDescriptionOpts = { useDefault?: boolean };

export const getDescription = (data: Order, opts?: GetDescriptionOpts) => {
  const {
    service: { i18nKeyName },
  } = data;

  switch (i18nKeyName) {
    case 'drive':
      return getDriveDescription(data);

    case 'taxi':
      return getGoDescription(data);

    case 'help_nearby':
      return i18n('Спасибо, что помогаете!');

    default:
      return opts?.useDefault ? getDefaultDescription(data) : undefined;
  }
};
